/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.io.gff;

/**
 * A factory to serve instances of GFF helper classes
 */
public class GffHelperFactory
{

  /**
   * Returns a class to process the GFF line based on inspecting its column
   * data. This may return a general-purpose GFF2 or GFF3 helper, or a
   * specialisation for a flavour of GFF generated by a particular tool.
   * 
   * @param gff
   * @return
   */
  public static GffHelperI getHelper(String[] gff)
  {
    if (gff == null || gff.length < 6)
    {
      return null;
    }

    GffHelperI result = null;
    if (ExonerateHelper.recognises(gff))
    {
      result = new ExonerateHelper();
    }
    else if (InterProScanHelper.recognises(gff))
    {
      result = new InterProScanHelper();
    }
    else if (looksLikeGff3(gff))
    {
      result = new Gff3Helper();
    }
    else
    {
      result = new Gff2Helper();
    }

    return result;
  }

  /**
   * Heuristic rule: if column 9 seems to have Name=Value entries, assume this
   * is GFF3. GFF3 uses '=' as name-value separator, GFF2 uses space ' '.
   * 
   * @param gff
   * @return
   */
  protected static boolean looksLikeGff3(String[] gff)
  {
    if (gff.length >= 9)
    {
      String attributes = gff[8].trim();
      int pos1 = attributes.indexOf(';');
      int pos2 = attributes.indexOf('=');
      if (pos2 != -1 && (pos1 == -1 || pos2 < pos1))
      {
        // there is an '=' before the first ';' (if any)
        // not foolproof as theoretically GFF2 could be like "Name Value=123;"
        return true;
      }
    }
    return false;
  }

}
