#ifndef C4CORE_SINGLE_HEADER
#include <stdint.h>
#include <c4/substr.hpp>
#endif
#include <iostream>

namespace c4 {

template<class T>
struct overflow64case
{
    bool is_overflow;
    T wrapped;
    csubstr dec;
    csubstr hex;
    csubstr oct;
    csubstr bin;
};

template<class T>
struct overflow64cases;

#define INT64_1(v) INT64_C(v) - INT64_C(1)  // the min value is not representable!

#define oc(T, is_overflow, wrapped, dec, hex, oct, bin)                 \
    overflow64case<T>{is_overflow, wrapped, csubstr{dec}, csubstr{hex}, csubstr{oct}, csubstr{bin}}

template<>
struct overflow64cases<int64_t>
{
    static constexpr const overflow64case<int64_t> values[] = {
        oc(int64_t, true , INT64_C( 9223372036854775803), "-9223372036854775813", "-0x8000000000000005", "-0o1000000000000000000005", "-0b1000000000000000000000000000000000000000000000000000000000000101"),
        oc(int64_t, true , INT64_C( 9223372036854775804), "-9223372036854775812", "-0x8000000000000004", "-0o1000000000000000000004", "-0b1000000000000000000000000000000000000000000000000000000000000100"),
        oc(int64_t, true , INT64_C( 9223372036854775805), "-9223372036854775811", "-0x8000000000000003", "-0o1000000000000000000003", "-0b1000000000000000000000000000000000000000000000000000000000000011"),
        oc(int64_t, true , INT64_C( 9223372036854775806), "-9223372036854775810", "-0x8000000000000002", "-0o1000000000000000000002", "-0b1000000000000000000000000000000000000000000000000000000000000010"),
        oc(int64_t, true , INT64_C( 9223372036854775807), "-9223372036854775809", "-0x8000000000000001", "-0o1000000000000000000001", "-0b1000000000000000000000000000000000000000000000000000000000000001"),
        oc(int64_t, false, INT64_1(-9223372036854775807), "-9223372036854775808", "-0x8000000000000000", "-0o1000000000000000000000", "-0b1000000000000000000000000000000000000000000000000000000000000000"),
        oc(int64_t, false, INT64_C(-9223372036854775807), "-9223372036854775807", "-0x7fffffffffffffff", "-0o777777777777777777777" , "-0b111111111111111111111111111111111111111111111111111111111111111"),
        oc(int64_t, false, INT64_C(-9223372036854775806), "-9223372036854775806", "-0x7ffffffffffffffe", "-0o777777777777777777776" , "-0b111111111111111111111111111111111111111111111111111111111111110"),
        oc(int64_t, false, INT64_C(-9223372036854775805), "-9223372036854775805", "-0x7ffffffffffffffd", "-0o777777777777777777775" , "-0b111111111111111111111111111111111111111111111111111111111111101"),
        oc(int64_t, false, INT64_C( 9223372036854775804),  "9223372036854775804",  "0x7ffffffffffffffc",  "0o777777777777777777774" ,  "0b111111111111111111111111111111111111111111111111111111111111100"),
        oc(int64_t, false, INT64_C( 9223372036854775805),  "9223372036854775805",  "0x7ffffffffffffffd",  "0o777777777777777777775" ,  "0b111111111111111111111111111111111111111111111111111111111111101"),
        oc(int64_t, false, INT64_C( 9223372036854775806),  "9223372036854775806",  "0x7ffffffffffffffe",  "0o777777777777777777776" ,  "0b111111111111111111111111111111111111111111111111111111111111110"),
        oc(int64_t, false, INT64_C( 9223372036854775807),  "9223372036854775807",  "0x7fffffffffffffff",  "0o777777777777777777777" ,  "0b111111111111111111111111111111111111111111111111111111111111111"),
        oc(int64_t, true , INT64_1(-9223372036854775807),  "9223372036854775808",  "0x8000000000000000",  "0o1000000000000000000000",  "0b1000000000000000000000000000000000000000000000000000000000000000"),
        oc(int64_t, true , INT64_C(-9223372036854775807),  "9223372036854775809",  "0x8000000000000001",  "0o1000000000000000000001",  "0b1000000000000000000000000000000000000000000000000000000000000001"),
        oc(int64_t, true , INT64_C(-9223372036854775806),  "9223372036854775810",  "0x8000000000000002",  "0o1000000000000000000002",  "0b1000000000000000000000000000000000000000000000000000000000000010"),
        oc(int64_t, true , INT64_C(-9223372036854775805),  "9223372036854775811",  "0x8000000000000003",  "0o1000000000000000000003",  "0b1000000000000000000000000000000000000000000000000000000000000011"),
        oc(int64_t, true , INT64_C(-9223372036854775804),  "9223372036854775812",  "0x8000000000000004",  "0o1000000000000000000004",  "0b1000000000000000000000000000000000000000000000000000000000000100"),
    };
};

template<>
struct overflow64cases<uint64_t>
{
    static constexpr const overflow64case<uint64_t> values[] = {
        oc(uint64_t, true , UINT64_C(18446744073709551611), "-5"                  , "-0x5"               , "-0o5"                     , "-0b101"),
        oc(uint64_t, true , UINT64_C(18446744073709551612), "-4"                  , "-0x4"               , "-0o4"                     , "-0b100"),
        oc(uint64_t, true , UINT64_C(18446744073709551613), "-3"                  , "-0x3"               , "-0o3"                     , "-0b11"),
        oc(uint64_t, true , UINT64_C(18446744073709551614), "-2"                  , "-0x2"               , "-0o2"                     , "-0b10"),
        oc(uint64_t, true , UINT64_C(18446744073709551615), "-1"                  , "-0x1"               , "-0o1"                     , "-0b1"),
        oc(uint64_t, false, UINT64_C(                   0), "0"                   , "0x0"                , "0o0"                      , "0b0"),
        oc(uint64_t, false, UINT64_C(                   1), "1"                   , "0x1"                , "0o1"                      , "0b1"),
        oc(uint64_t, false, UINT64_C(                   2), "2"                   , "0x2"                , "0o2"                      , "0b10"),
        oc(uint64_t, false, UINT64_C(                   3), "3"                   , "0x3"                , "0o3"                      , "0b11"),
        oc(uint64_t, false, UINT64_C(18446744073709551612), "18446744073709551612", "0xfffffffffffffffc" , "0o1777777777777777777774" , "0b1111111111111111111111111111111111111111111111111111111111111100"),
        oc(uint64_t, false, UINT64_C(18446744073709551613), "18446744073709551613", "0xfffffffffffffffd" , "0o1777777777777777777775" , "0b1111111111111111111111111111111111111111111111111111111111111101"),
        oc(uint64_t, false, UINT64_C(18446744073709551614), "18446744073709551614", "0xfffffffffffffffe" , "0o1777777777777777777776" , "0b1111111111111111111111111111111111111111111111111111111111111110"),
        oc(uint64_t, false, UINT64_C(18446744073709551615), "18446744073709551615", "0xffffffffffffffff" , "0o1777777777777777777777" , "0b1111111111111111111111111111111111111111111111111111111111111111"),
        oc(uint64_t, true , UINT64_C(                   0), "18446744073709551616", "0x10000000000000000", "0o20000000000000000000000", "0b10000000000000000000000000000000000000000000000000000000000000000"),
        oc(uint64_t, true , UINT64_C(                   1), "18446744073709551617", "0x10000000000000001", "0o20000000000000000000001", "0b10000000000000000000000000000000000000000000000000000000000000001"),
        oc(uint64_t, true , UINT64_C(                   2), "18446744073709551618", "0x10000000000000002", "0o20000000000000000000002", "0b10000000000000000000000000000000000000000000000000000000000000010"),
        oc(uint64_t, true , UINT64_C(                   3), "18446744073709551619", "0x10000000000000003", "0o20000000000000000000003", "0b10000000000000000000000000000000000000000000000000000000000000011"),
        oc(uint64_t, true , UINT64_C(                   4), "18446744073709551620", "0x10000000000000004", "0o20000000000000000000004", "0b10000000000000000000000000000000000000000000000000000000000000100"),
        oc(uint64_t, true , UINT64_C(                   5), "18446744073709551621", "0x10000000000000005", "0o20000000000000000000005", "0b10000000000000000000000000000000000000000000000000000000000000101"),
    };
};

constexpr const overflow64case<int64_t> overflow64cases<int64_t>::values[];
constexpr const overflow64case<uint64_t> overflow64cases<uint64_t>::values[];

#undef oc


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

struct invalid_case
{
    csubstr dec, hex, oct, bin;
};

C4_INLINE_CONSTEXPR const invalid_case invalid_cases[] = {
#define ic(dec, hex, oct, bin)                                          \
    invalid_case{csubstr{dec}, csubstr{hex}, csubstr{oct}, csubstr{bin}}
    ic(""   , ""     , ""     , ""),
    ic(" "  , " "    , " "    , " "),
    ic("."  , "."    , "."    , "."),
    ic("-"  , "-"    , "-"    , "-"),
    ic("\t" , "\t"   , "\t"   , "\t"),
    ic("\n" , "\n"   , "\n"   , "\n"),
    ic("...", "..."  , "..."  , "..."),
    ic("===", "==="  , "==="  , "==="),
    ic("=??", "???"  , "???"  , "???"),
    ic("12a", "0x12g", "0o128", "0b102"),
    ic("0.1", "0x1.2", "0o1.2", "0b1.1"),
    ic("0,1", "0x1,2", "0o1,2", "0b1,1"),
    ic("zz" , "0xzz" , "0ozz" , "0bzz"),
    ic(""   , "0x"   , "0o"   , "0b"),
    ic("- " , "-0x"  , "-0o"  , "-0b"),
    ic("2 0", "0x2 0", "0o2 0", "0b1 0"),
    ic(" 2" , " 0x2" , " 0o2" , " 0b1"),
    ic("\t2", "\t0x2", "\t0o2", "\t0b1"),
    ic("\n2", "nt0x2", "\n0o2", "\n0b1"),
    ic("2 " , "0x2 " , "0o2 " , "0b1 "),
    ic("2\t", "0x2\t", "0o2\t", "0b1\t"),
    ic("2\n", "0x2\n", "0o2\n", "0b1\n"),
    ic("nan", "nan", "nan", "nan"),
    ic("NaN", "NaN", "NaN", "NaN"),
    ic("Inf", "Inf", "Inf", "Inf"),
    ic("inf", "inf", "inf", "inf"),
    ic("infinity", "infinity", "infinity", "infinity"),
    ic("Infinity", "Infinity", "Infinity", "Infinity"),
    ic("somevalue", "somevalue", "somevalue", "somevalue"),
    ic("2345kjhiuy3245", "2345kjhiuy3245", "2345kjhiuy3245", "2345kjhiuy3245"),
#undef ic
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

template<class T>
struct number_case
{
    T val;
    csubstr dec;
    csubstr hex;
    csubstr oct;
    csubstr bin;
    T valp1;
    T has_valp1;
    constexpr number_case() noexcept = default;
    constexpr number_case(T val_, csubstr dec_, csubstr hex_, csubstr oct_, csubstr bin_, T valp1_) noexcept
        : val(val_)
        , dec(dec_)
        , hex(hex_)
        , oct(oct_)
        , bin(bin_)
        , valp1(valp1_)
        , has_valp1(true)
    {
    }
    constexpr number_case(T val_, csubstr dec_, csubstr hex_, csubstr oct_, csubstr bin_) noexcept
        : val(val_)
        , dec(dec_)
        , hex(hex_)
        , oct(oct_)
        , bin(bin_)
        , valp1()
        , has_valp1(false)
    {
    }
};

template<class T>
std::ostream& operator<< (std::ostream& s, number_case<T> const& nc)
{
    using upcast_t = typename std::conditional<sizeof(T)!=1, T,
                                               typename std::conditional<std::is_signed<T>::value,
                                                                         int64_t,
                                                                         uint64_t>::type>::type;
    s << "val=" << (upcast_t)nc.val << "  "
      << "dec=" << nc.dec << "  "
      << "hex=" << nc.hex << "  "
      << "oct=" << nc.oct << "  "
      << "bin=" << nc.bin;
    return s;
}

template<class T> struct numbers;

#define ITER_NUMBERS(ty, number_name) for(number_case<ty> const& C4_RESTRICT number_name : numbers<ty>::vals)

C4_SUPPRESS_WARNING_MSVC_PUSH
C4_SUPPRESS_WARNING_MSVC(4146)
C4_SUPPRESS_WARNING_MSVC(4305)
C4_SUPPRESS_WARNING_MSVC(4310)

// these numbers were generated with printintegers.py, in this dir.

#define ncp__(tyval, val, dec, hex, bin, oct, tyvalp1, valp1) number_case<value_type>((value_type)(tyval(val)), csubstr{dec}, csubstr{hex}, csubstr{bin}, csubstr{oct}, (value_type)(tyvalp1(valp1)))
#define ncp_(ty, val, dec, hex, bin, oct, valp1) number_case<value_type>((value_type)(ty(val)), csubstr{dec}, csubstr{hex}, csubstr{bin}, csubstr{oct}, (value_type)(ty(valp1)))
#define nc_(ty, val, dec, hex, bin, oct) number_case<value_type>((value_type)(ty(val)), csubstr{dec}, csubstr{hex}, csubstr{bin}, csubstr{oct})

template<>
struct numbers<int8_t>
{
    using value_type = int8_t;
    static C4_INLINE_CONSTEXPR const number_case<int8_t> vals[] = {
        #define nc(val, dec, hex, bin, oct, valp1) ncp_(INT8_C, val, dec, hex, bin, oct, valp1)
        nc(-128, "-128", "-0x80", "-0o200", "-0b10000000", -127),
        nc(-127, "-127", "-0x7f", "-0o177", "-0b1111111", -126),
        nc(-126, "-126", "-0x7e", "-0o176", "-0b1111110", -125),
        nc(-125, "-125", "-0x7d", "-0o175", "-0b1111101", -124),
        nc(-124, "-124", "-0x7c", "-0o174", "-0b1111100", -123),
        nc(-123, "-123", "-0x7b", "-0o173", "-0b1111011", -122),
        nc(-101, "-101", "-0x65", "-0o145", "-0b1100101", -100),
        nc(-100, "-100", "-0x64", "-0o144", "-0b1100100", -99),
        nc(-99, "-99", "-0x63", "-0o143", "-0b1100011", -98),
        nc(-65, "-65", "-0x41", "-0o101", "-0b1000001", -64),
        nc(-64, "-64", "-0x40", "-0o100", "-0b1000000", -63),
        nc(-63, "-63", "-0x3f", "-0o77", "-0b111111", -62),
        nc(-33, "-33", "-0x21", "-0o41", "-0b100001", -32),
        nc(-32, "-32", "-0x20", "-0o40", "-0b100000", -31),
        nc(-31, "-31", "-0x1f", "-0o37", "-0b11111", -30),
        nc(-17, "-17", "-0x11", "-0o21", "-0b10001", -16),
        nc(-16, "-16", "-0x10", "-0o20", "-0b10000", -15),
        nc(-15, "-15", "-0xf", "-0o17", "-0b1111", -14),
        nc(-12, "-12", "-0xc", "-0o14", "-0b1100", -11),
        nc(-11, "-11", "-0xb", "-0o13", "-0b1011", -10),
        nc(-10, "-10", "-0xa", "-0o12", "-0b1010", -9),
        nc(-9, "-9", "-0x9", "-0o11", "-0b1001", -8),
        nc(-8, "-8", "-0x8", "-0o10", "-0b1000", -7),
        nc(-7, "-7", "-0x7", "-0o7", "-0b111", -6),
        nc(-6, "-6", "-0x6", "-0o6", "-0b110", -5),
        nc(-5, "-5", "-0x5", "-0o5", "-0b101", -4),
        nc(-4, "-4", "-0x4", "-0o4", "-0b100", -3),
        nc(-3, "-3", "-0x3", "-0o3", "-0b11", -2),
        nc(-2, "-2", "-0x2", "-0o2", "-0b10", -1),
        nc(-1, "-1", "-0x1", "-0o1", "-0b1", 0),
        nc(0, "0", "0x0", "0o0", "0b0", 1),
        nc(1, "1", "0x1", "0o1", "0b1", 2),
        nc(2, "2", "0x2", "0o2", "0b10", 3),
        nc(3, "3", "0x3", "0o3", "0b11", 4),
        nc(4, "4", "0x4", "0o4", "0b100", 5),
        nc(5, "5", "0x5", "0o5", "0b101", 6),
        nc(6, "6", "0x6", "0o6", "0b110", 7),
        nc(7, "7", "0x7", "0o7", "0b111", 8),
        nc(8, "8", "0x8", "0o10", "0b1000", 9),
        nc(9, "9", "0x9", "0o11", "0b1001", 10),
        nc(10, "10", "0xa", "0o12", "0b1010", 11),
        nc(11, "11", "0xb", "0o13", "0b1011", 12),
        nc(12, "12", "0xc", "0o14", "0b1100", 13),
        nc(13, "13", "0xd", "0o15", "0b1101", 14),
        nc(15, "15", "0xf", "0o17", "0b1111", 16),
        nc(16, "16", "0x10", "0o20", "0b10000", 17),
        nc(17, "17", "0x11", "0o21", "0b10001", 18),
        nc(31, "31", "0x1f", "0o37", "0b11111", 32),
        nc(32, "32", "0x20", "0o40", "0b100000", 33),
        nc(33, "33", "0x21", "0o41", "0b100001", 34),
        nc(63, "63", "0x3f", "0o77", "0b111111", 64),
        nc(64, "64", "0x40", "0o100", "0b1000000", 65),
        nc(65, "65", "0x41", "0o101", "0b1000001", 66),
        nc(99, "99", "0x63", "0o143", "0b1100011", 100),
        nc(100, "100", "0x64", "0o144", "0b1100100", 101),
        nc(101, "101", "0x65", "0o145", "0b1100101", 102),
        nc(122, "122", "0x7a", "0o172", "0b1111010", 123),
        nc(123, "123", "0x7b", "0o173", "0b1111011", 124),
        nc(124, "124", "0x7c", "0o174", "0b1111100", 125),
        nc(125, "125", "0x7d", "0o175", "0b1111101", 126),
        nc(126, "126", "0x7e", "0o176", "0b1111110", 127),
        nc(127, "127", "0x7f", "0o177", "0b1111111", -128),
        #undef nc
    };
};


template<>
struct numbers<uint8_t>
{
    using value_type = uint8_t;
    static C4_INLINE_CONSTEXPR const number_case<uint8_t> vals[] = {
        #define nc(val, dec, hex, bin, oct, valp1) ncp_(UINT8_C, val, dec, hex, bin, oct, valp1)
        nc(0, "0", "0x0", "0o0", "0b0", 1),
        nc(1, "1", "0x1", "0o1", "0b1", 2),
        nc(2, "2", "0x2", "0o2", "0b10", 3),
        nc(3, "3", "0x3", "0o3", "0b11", 4),
        nc(4, "4", "0x4", "0o4", "0b100", 5),
        nc(5, "5", "0x5", "0o5", "0b101", 6),
        nc(6, "6", "0x6", "0o6", "0b110", 7),
        nc(7, "7", "0x7", "0o7", "0b111", 8),
        nc(8, "8", "0x8", "0o10", "0b1000", 9),
        nc(9, "9", "0x9", "0o11", "0b1001", 10),
        nc(10, "10", "0xa", "0o12", "0b1010", 11),
        nc(11, "11", "0xb", "0o13", "0b1011", 12),
        nc(15, "15", "0xf", "0o17", "0b1111", 16),
        nc(16, "16", "0x10", "0o20", "0b10000", 17),
        nc(17, "17", "0x11", "0o21", "0b10001", 18),
        nc(24, "24", "0x18", "0o30", "0b11000", 25),
        nc(25, "25", "0x19", "0o31", "0b11001", 26),
        nc(26, "26", "0x1a", "0o32", "0b11010", 27),
        nc(31, "31", "0x1f", "0o37", "0b11111", 32),
        nc(32, "32", "0x20", "0o40", "0b100000", 33),
        nc(33, "33", "0x21", "0o41", "0b100001", 34),
        nc(63, "63", "0x3f", "0o77", "0b111111", 64),
        nc(64, "64", "0x40", "0o100", "0b1000000", 65),
        nc(65, "65", "0x41", "0o101", "0b1000001", 66),
        nc(99, "99", "0x63", "0o143", "0b1100011", 100),
        nc(100, "100", "0x64", "0o144", "0b1100100", 101),
        nc(101, "101", "0x65", "0o145", "0b1100101", 102),
        nc(127, "127", "0x7f", "0o177", "0b1111111", 128),
        nc(128, "128", "0x80", "0o200", "0b10000000", 129),
        nc(129, "129", "0x81", "0o201", "0b10000001", 130),
        nc(251, "251", "0xfb", "0o373", "0b11111011", 252),
        nc(252, "252", "0xfc", "0o374", "0b11111100", 253),
        nc(253, "253", "0xfd", "0o375", "0b11111101", 254),
        nc(254, "254", "0xfe", "0o376", "0b11111110", 255),
        nc(255, "255", "0xff", "0o377", "0b11111111", 0),
        #undef nc
    };
};


template<>
struct numbers<int16_t>
{
    using value_type = int16_t;
    static C4_INLINE_CONSTEXPR const number_case<int16_t> vals[] = {
        #define nc(val, dec, hex, bin, oct) nc_(INT16_C, val, dec, hex, bin, oct)
        #define pc(val, dec, hex, bin, oct, valp1) ncp_(INT16_C, val, dec, hex, bin, oct, valp1)
        pc(-32768, "-32768", "-0x8000", "-0o100000", "-0b1000000000000000", -32767),
        pc(-32767, "-32767", "-0x7fff", "-0o77777", "-0b111111111111111", -32766),
        pc(-32766, "-32766", "-0x7ffe", "-0o77776", "-0b111111111111110", -32765),
        pc(-32765, "-32765", "-0x7ffd", "-0o77775", "-0b111111111111101", -32764),
        pc(-32764, "-32764", "-0x7ffc", "-0o77774", "-0b111111111111100", -32763),
        pc(-32763, "-32763", "-0x7ffb", "-0o77773", "-0b111111111111011", -32762),
        pc(-16385, "-16385", "-0x4001", "-0o40001", "-0b100000000000001", -16384),
        pc(-16384, "-16384", "-0x4000", "-0o40000", "-0b100000000000000", -16383),
        pc(-16383, "-16383", "-0x3fff", "-0o37777", "-0b11111111111111", -16382),
        pc(-10001, "-10001", "-0x2711", "-0o23421", "-0b10011100010001", -10000),
        pc(-10000, "-10000", "-0x2710", "-0o23420", "-0b10011100010000", -9999),
        pc(-9999, "-9999", "-0x270f", "-0o23417", "-0b10011100001111", -9998),
        pc(-8193, "-8193", "-0x2001", "-0o20001", "-0b10000000000001", -8192),
        pc(-8192, "-8192", "-0x2000", "-0o20000", "-0b10000000000000", -8191),
        nc(-8191, "-8191", "-0x1fff", "-0o17777", "-0b1111111111111"),
        nc(-4097, "-4097", "-0x1001", "-0o10001", "-0b1000000000001"),
        nc(-4096, "-4096", "-0x1000", "-0o10000", "-0b1000000000000"),
        nc(-4095, "-4095", "-0xfff", "-0o7777", "-0b111111111111"),
        nc(-3276, "-3276", "-0xccc", "-0o6314", "-0b110011001100"),
        nc(-2049, "-2049", "-0x801", "-0o4001", "-0b100000000001"),
        nc(-2048, "-2048", "-0x800", "-0o4000", "-0b100000000000"),
        nc(-2047, "-2047", "-0x7ff", "-0o3777", "-0b11111111111"),
        nc(-1025, "-1025", "-0x401", "-0o2001", "-0b10000000001"),
        nc(-1024, "-1024", "-0x400", "-0o2000", "-0b10000000000"),
        nc(-1023, "-1023", "-0x3ff", "-0o1777", "-0b1111111111"),
        nc(-1001, "-1001", "-0x3e9", "-0o1751", "-0b1111101001"),
        nc(-1000, "-1000", "-0x3e8", "-0o1750", "-0b1111101000"),
        nc(-999, "-999", "-0x3e7", "-0o1747", "-0b1111100111"),
        nc(-513, "-513", "-0x201", "-0o1001", "-0b1000000001"),
        nc(-512, "-512", "-0x200", "-0o1000", "-0b1000000000"),
        nc(-511, "-511", "-0x1ff", "-0o777", "-0b111111111"),
        nc(-327, "-327", "-0x147", "-0o507", "-0b101000111"),
        nc(-257, "-257", "-0x101", "-0o401", "-0b100000001"),
        nc(-256, "-256", "-0x100", "-0o400", "-0b100000000"),
        nc(-255, "-255", "-0xff", "-0o377", "-0b11111111"),
        nc(-129, "-129", "-0x81", "-0o201", "-0b10000001"),
        nc(-128, "-128", "-0x80", "-0o200", "-0b10000000"),
        nc(-127, "-127", "-0x7f", "-0o177", "-0b1111111"),
        nc(-101, "-101", "-0x65", "-0o145", "-0b1100101"),
        nc(-100, "-100", "-0x64", "-0o144", "-0b1100100"),
        nc(-99, "-99", "-0x63", "-0o143", "-0b1100011"),
        nc(-65, "-65", "-0x41", "-0o101", "-0b1000001"),
        nc(-64, "-64", "-0x40", "-0o100", "-0b1000000"),
        nc(-63, "-63", "-0x3f", "-0o77", "-0b111111"),
        nc(-33, "-33", "-0x21", "-0o41", "-0b100001"),
        nc(-32, "-32", "-0x20", "-0o40", "-0b100000"),
        nc(-31, "-31", "-0x1f", "-0o37", "-0b11111"),
        nc(-17, "-17", "-0x11", "-0o21", "-0b10001"),
        nc(-16, "-16", "-0x10", "-0o20", "-0b10000"),
        nc(-15, "-15", "-0xf", "-0o17", "-0b1111"),
        nc(-11, "-11", "-0xb", "-0o13", "-0b1011"),
        nc(-10, "-10", "-0xa", "-0o12", "-0b1010"),
        nc(-9, "-9", "-0x9", "-0o11", "-0b1001"),
        nc(-8, "-8", "-0x8", "-0o10", "-0b1000"),
        nc(-7, "-7", "-0x7", "-0o7", "-0b111"),
        nc(-6, "-6", "-0x6", "-0o6", "-0b110"),
        nc(-5, "-5", "-0x5", "-0o5", "-0b101"),
        nc(-4, "-4", "-0x4", "-0o4", "-0b100"),
        nc(-3, "-3", "-0x3", "-0o3", "-0b11"),
        nc(-2, "-2", "-0x2", "-0o2", "-0b10"),
        nc(-1, "-1", "-0x1", "-0o1", "-0b1"),
        nc(0, "0", "0x0", "0o0", "0b0"),
        nc(1, "1", "0x1", "0o1", "0b1"),
        nc(2, "2", "0x2", "0o2", "0b10"),
        nc(3, "3", "0x3", "0o3", "0b11"),
        nc(4, "4", "0x4", "0o4", "0b100"),
        nc(5, "5", "0x5", "0o5", "0b101"),
        nc(6, "6", "0x6", "0o6", "0b110"),
        nc(7, "7", "0x7", "0o7", "0b111"),
        nc(8, "8", "0x8", "0o10", "0b1000"),
        nc(9, "9", "0x9", "0o11", "0b1001"),
        nc(10, "10", "0xa", "0o12", "0b1010"),
        nc(11, "11", "0xb", "0o13", "0b1011"),
        nc(15, "15", "0xf", "0o17", "0b1111"),
        nc(16, "16", "0x10", "0o20", "0b10000"),
        nc(17, "17", "0x11", "0o21", "0b10001"),
        nc(31, "31", "0x1f", "0o37", "0b11111"),
        nc(32, "32", "0x20", "0o40", "0b100000"),
        nc(33, "33", "0x21", "0o41", "0b100001"),
        nc(63, "63", "0x3f", "0o77", "0b111111"),
        nc(64, "64", "0x40", "0o100", "0b1000000"),
        nc(65, "65", "0x41", "0o101", "0b1000001"),
        nc(99, "99", "0x63", "0o143", "0b1100011"),
        nc(100, "100", "0x64", "0o144", "0b1100100"),
        nc(101, "101", "0x65", "0o145", "0b1100101"),
        nc(127, "127", "0x7f", "0o177", "0b1111111"),
        nc(128, "128", "0x80", "0o200", "0b10000000"),
        nc(129, "129", "0x81", "0o201", "0b10000001"),
        nc(255, "255", "0xff", "0o377", "0b11111111"),
        nc(256, "256", "0x100", "0o400", "0b100000000"),
        nc(257, "257", "0x101", "0o401", "0b100000001"),
        nc(326, "326", "0x146", "0o506", "0b101000110"),
        nc(327, "327", "0x147", "0o507", "0b101000111"),
        nc(328, "328", "0x148", "0o510", "0b101001000"),
        nc(511, "511", "0x1ff", "0o777", "0b111111111"),
        nc(512, "512", "0x200", "0o1000", "0b1000000000"),
        nc(513, "513", "0x201", "0o1001", "0b1000000001"),
        nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
        nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
        nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
        nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
        nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
        nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
        nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
        nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
        nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
        nc(3275, "3275", "0xccb", "0o6313", "0b110011001011"),
        nc(3276, "3276", "0xccc", "0o6314", "0b110011001100"),
        nc(3277, "3277", "0xccd", "0o6315", "0b110011001101"),
        nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
        nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
        nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
        nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
        nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
        nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
        nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
        nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
        nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
        nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
        nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
        nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
        nc(32762, "32762", "0x7ffa", "0o77772", "0b111111111111010"),
        nc(32763, "32763", "0x7ffb", "0o77773", "0b111111111111011"),
        nc(32764, "32764", "0x7ffc", "0o77774", "0b111111111111100"),
        nc(32765, "32765", "0x7ffd", "0o77775", "0b111111111111101"),
        pc(32766, "32766", "0x7ffe", "0o77776", "0b111111111111110", 32767),
        pc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111", -32768),
        #undef nc
        #undef pc
    };
};


template<>
struct numbers<uint16_t>
{
    using value_type = uint16_t;
    static C4_INLINE_CONSTEXPR const number_case<uint16_t> vals[] = {
         #define nc(val, dec, hex, bin, oct) nc_(UINT16_C, val, dec, hex, bin, oct)
         #define pc(val, dec, hex, bin, oct, valp1) ncp_(UINT16_C, val, dec, hex, bin, oct, valp1)
         pc(0, "0", "0x0", "0o0", "0b0", 1),
         nc(1, "1", "0x1", "0o1", "0b1"),
         nc(2, "2", "0x2", "0o2", "0b10"),
         nc(3, "3", "0x3", "0o3", "0b11"),
         nc(4, "4", "0x4", "0o4", "0b100"),
         nc(5, "5", "0x5", "0o5", "0b101"),
         nc(6, "6", "0x6", "0o6", "0b110"),
         nc(7, "7", "0x7", "0o7", "0b111"),
         nc(8, "8", "0x8", "0o10", "0b1000"),
         nc(9, "9", "0x9", "0o11", "0b1001"),
         nc(10, "10", "0xa", "0o12", "0b1010"),
         nc(11, "11", "0xb", "0o13", "0b1011"),
         nc(15, "15", "0xf", "0o17", "0b1111"),
         nc(16, "16", "0x10", "0o20", "0b10000"),
         nc(17, "17", "0x11", "0o21", "0b10001"),
         nc(31, "31", "0x1f", "0o37", "0b11111"),
         nc(32, "32", "0x20", "0o40", "0b100000"),
         nc(33, "33", "0x21", "0o41", "0b100001"),
         nc(63, "63", "0x3f", "0o77", "0b111111"),
         nc(64, "64", "0x40", "0o100", "0b1000000"),
         nc(65, "65", "0x41", "0o101", "0b1000001"),
         nc(66, "66", "0x42", "0o102", "0b1000010"),
         nc(99, "99", "0x63", "0o143", "0b1100011"),
         nc(100, "100", "0x64", "0o144", "0b1100100"),
         nc(101, "101", "0x65", "0o145", "0b1100101"),
         nc(127, "127", "0x7f", "0o177", "0b1111111"),
         nc(128, "128", "0x80", "0o200", "0b10000000"),
         nc(129, "129", "0x81", "0o201", "0b10000001"),
         nc(255, "255", "0xff", "0o377", "0b11111111"),
         nc(256, "256", "0x100", "0o400", "0b100000000"),
         nc(257, "257", "0x101", "0o401", "0b100000001"),
         nc(511, "511", "0x1ff", "0o777", "0b111111111"),
         nc(512, "512", "0x200", "0o1000", "0b1000000000"),
         nc(513, "513", "0x201", "0o1001", "0b1000000001"),
         nc(654, "654", "0x28e", "0o1216", "0b1010001110"),
         nc(655, "655", "0x28f", "0o1217", "0b1010001111"),
         nc(656, "656", "0x290", "0o1220", "0b1010010000"),
         nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
         nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
         nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
         nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
         nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
         nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
         nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
         nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
         nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
         nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
         nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
         nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
         nc(6552, "6552", "0x1998", "0o14630", "0b1100110011000"),
         nc(6553, "6553", "0x1999", "0o14631", "0b1100110011001"),
         nc(6554, "6554", "0x199a", "0o14632", "0b1100110011010"),
         nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
         nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
         nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
         nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
         nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
         nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
         nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
         nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
         nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
         nc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111"),
         nc(32768, "32768", "0x8000", "0o100000", "0b1000000000000000"),
         nc(32769, "32769", "0x8001", "0o100001", "0b1000000000000001"),
         nc(65531, "65531", "0xfffb", "0o177773", "0b1111111111111011"),
         nc(65532, "65532", "0xfffc", "0o177774", "0b1111111111111100"),
         nc(65533, "65533", "0xfffd", "0o177775", "0b1111111111111101"),
         pc(65534, "65534", "0xfffe", "0o177776", "0b1111111111111110", 65535),
         pc(65535, "65535", "0xffff", "0o177777", "0b1111111111111111", 0),
         #undef nc
         #undef pc
    };
};


template<>
struct numbers<int32_t>
{
    using value_type = int32_t;
    static C4_INLINE_CONSTEXPR const number_case<int32_t> vals[] = {
        #define nc(val, dec, hex, bin, oct) nc_(INT32_C, val, dec, hex, bin, oct)
        #define pc(val, dec, hex, bin, oct, valp1) ncp_(INT32_C, val, dec, hex, bin, oct, valp1)
        nc(-2147483648, "-2147483648", "-0x80000000", "-0o20000000000", "-0b10000000000000000000000000000000"),
        nc(-2147483647, "-2147483647", "-0x7fffffff", "-0o17777777777", "-0b1111111111111111111111111111111"),
        nc(-2147483646, "-2147483646", "-0x7ffffffe", "-0o17777777776", "-0b1111111111111111111111111111110"),
        nc(-2147483645, "-2147483645", "-0x7ffffffd", "-0o17777777775", "-0b1111111111111111111111111111101"),
        nc(-2147483644, "-2147483644", "-0x7ffffffc", "-0o17777777774", "-0b1111111111111111111111111111100"),
        nc(-2147483643, "-2147483643", "-0x7ffffffb", "-0o17777777773", "-0b1111111111111111111111111111011"),
        nc(-1073741825, "-1073741825", "-0x40000001", "-0o10000000001", "-0b1000000000000000000000000000001"),
        nc(-1073741824, "-1073741824", "-0x40000000", "-0o10000000000", "-0b1000000000000000000000000000000"),
        nc(-1073741823, "-1073741823", "-0x3fffffff", "-0o7777777777", "-0b111111111111111111111111111111"),
        nc(-1000000001, "-1000000001", "-0x3b9aca01", "-0o7346545001", "-0b111011100110101100101000000001"),
        nc(-1000000000, "-1000000000", "-0x3b9aca00", "-0o7346545000", "-0b111011100110101100101000000000"),
        nc(-999999999, "-999999999", "-0x3b9ac9ff", "-0o7346544777", "-0b111011100110101100100111111111"),
        nc(-536870913, "-536870913", "-0x20000001", "-0o4000000001", "-0b100000000000000000000000000001"),
        nc(-536870912, "-536870912", "-0x20000000", "-0o4000000000", "-0b100000000000000000000000000000"),
        nc(-536870911, "-536870911", "-0x1fffffff", "-0o3777777777", "-0b11111111111111111111111111111"),
        nc(-268435457, "-268435457", "-0x10000001", "-0o2000000001", "-0b10000000000000000000000000001"),
        nc(-268435456, "-268435456", "-0x10000000", "-0o2000000000", "-0b10000000000000000000000000000"),
        nc(-268435455, "-268435455", "-0xfffffff", "-0o1777777777", "-0b1111111111111111111111111111"),
        nc(-214748364, "-214748364", "-0xccccccc", "-0o1463146314", "-0b1100110011001100110011001100"),
        nc(-134217729, "-134217729", "-0x8000001", "-0o1000000001", "-0b1000000000000000000000000001"),
        nc(-134217728, "-134217728", "-0x8000000", "-0o1000000000", "-0b1000000000000000000000000000"),
        nc(-134217727, "-134217727", "-0x7ffffff", "-0o777777777", "-0b111111111111111111111111111"),
        nc(-100000001, "-100000001", "-0x5f5e101", "-0o575360401", "-0b101111101011110000100000001"),
        nc(-100000000, "-100000000", "-0x5f5e100", "-0o575360400", "-0b101111101011110000100000000"),
        nc(-99999999, "-99999999", "-0x5f5e0ff", "-0o575360377", "-0b101111101011110000011111111"),
        nc(-67108865, "-67108865", "-0x4000001", "-0o400000001", "-0b100000000000000000000000001"),
        nc(-67108864, "-67108864", "-0x4000000", "-0o400000000", "-0b100000000000000000000000000"),
        nc(-67108863, "-67108863", "-0x3ffffff", "-0o377777777", "-0b11111111111111111111111111"),
        nc(-33554433, "-33554433", "-0x2000001", "-0o200000001", "-0b10000000000000000000000001"),
        nc(-33554432, "-33554432", "-0x2000000", "-0o200000000", "-0b10000000000000000000000000"),
        nc(-33554431, "-33554431", "-0x1ffffff", "-0o177777777", "-0b1111111111111111111111111"),
        nc(-21474836, "-21474836", "-0x147ae14", "-0o121727024", "-0b1010001111010111000010100"),
        nc(-16777217, "-16777217", "-0x1000001", "-0o100000001", "-0b1000000000000000000000001"),
        nc(-16777216, "-16777216", "-0x1000000", "-0o100000000", "-0b1000000000000000000000000"),
        nc(-16777215, "-16777215", "-0xffffff", "-0o77777777", "-0b111111111111111111111111"),
        nc(-10000001, "-10000001", "-0x989681", "-0o46113201", "-0b100110001001011010000001"),
        nc(-10000000, "-10000000", "-0x989680", "-0o46113200", "-0b100110001001011010000000"),
        nc(-9999999, "-9999999", "-0x98967f", "-0o46113177", "-0b100110001001011001111111"),
        nc(-8388609, "-8388609", "-0x800001", "-0o40000001", "-0b100000000000000000000001"),
        nc(-8388608, "-8388608", "-0x800000", "-0o40000000", "-0b100000000000000000000000"),
        nc(-8388607, "-8388607", "-0x7fffff", "-0o37777777", "-0b11111111111111111111111"),
        nc(-4194305, "-4194305", "-0x400001", "-0o20000001", "-0b10000000000000000000001"),
        nc(-4194304, "-4194304", "-0x400000", "-0o20000000", "-0b10000000000000000000000"),
        nc(-4194303, "-4194303", "-0x3fffff", "-0o17777777", "-0b1111111111111111111111"),
        nc(-2147483, "-2147483", "-0x20c49b", "-0o10142233", "-0b1000001100010010011011"),
        nc(-2097153, "-2097153", "-0x200001", "-0o10000001", "-0b1000000000000000000001"),
        nc(-2097152, "-2097152", "-0x200000", "-0o10000000", "-0b1000000000000000000000"),
        nc(-2097151, "-2097151", "-0x1fffff", "-0o7777777", "-0b111111111111111111111"),
        nc(-1048577, "-1048577", "-0x100001", "-0o4000001", "-0b100000000000000000001"),
        nc(-1048576, "-1048576", "-0x100000", "-0o4000000", "-0b100000000000000000000"),
        nc(-1048575, "-1048575", "-0xfffff", "-0o3777777", "-0b11111111111111111111"),
        nc(-1000001, "-1000001", "-0xf4241", "-0o3641101", "-0b11110100001001000001"),
        nc(-1000000, "-1000000", "-0xf4240", "-0o3641100", "-0b11110100001001000000"),
        nc(-999999, "-999999", "-0xf423f", "-0o3641077", "-0b11110100001000111111"),
        nc(-524289, "-524289", "-0x80001", "-0o2000001", "-0b10000000000000000001"),
        nc(-524288, "-524288", "-0x80000", "-0o2000000", "-0b10000000000000000000"),
        nc(-524287, "-524287", "-0x7ffff", "-0o1777777", "-0b1111111111111111111"),
        nc(-262145, "-262145", "-0x40001", "-0o1000001", "-0b1000000000000000001"),
        nc(-262144, "-262144", "-0x40000", "-0o1000000", "-0b1000000000000000000"),
        nc(-262143, "-262143", "-0x3ffff", "-0o777777", "-0b111111111111111111"),
        nc(-214748, "-214748", "-0x346dc", "-0o643334", "-0b110100011011011100"),
        nc(-131073, "-131073", "-0x20001", "-0o400001", "-0b100000000000000001"),
        nc(-131072, "-131072", "-0x20000", "-0o400000", "-0b100000000000000000"),
        nc(-131071, "-131071", "-0x1ffff", "-0o377777", "-0b11111111111111111"),
        nc(-100001, "-100001", "-0x186a1", "-0o303241", "-0b11000011010100001"),
        nc(-100000, "-100000", "-0x186a0", "-0o303240", "-0b11000011010100000"),
        nc(-99999, "-99999", "-0x1869f", "-0o303237", "-0b11000011010011111"),
        nc(-65537, "-65537", "-0x10001", "-0o200001", "-0b10000000000000001"),
        nc(-65536, "-65536", "-0x10000", "-0o200000", "-0b10000000000000000"),
        nc(-65535, "-65535", "-0xffff", "-0o177777", "-0b1111111111111111"),
        nc(-32769, "-32769", "-0x8001", "-0o100001", "-0b1000000000000001"),
        nc(-32768, "-32768", "-0x8000", "-0o100000", "-0b1000000000000000"),
        nc(-32767, "-32767", "-0x7fff", "-0o77777", "-0b111111111111111"),
        nc(-21474, "-21474", "-0x53e2", "-0o51742", "-0b101001111100010"),
        nc(-16385, "-16385", "-0x4001", "-0o40001", "-0b100000000000001"),
        nc(-16384, "-16384", "-0x4000", "-0o40000", "-0b100000000000000"),
        nc(-16383, "-16383", "-0x3fff", "-0o37777", "-0b11111111111111"),
        nc(-10001, "-10001", "-0x2711", "-0o23421", "-0b10011100010001"),
        nc(-10000, "-10000", "-0x2710", "-0o23420", "-0b10011100010000"),
        nc(-9999, "-9999", "-0x270f", "-0o23417", "-0b10011100001111"),
        nc(-8193, "-8193", "-0x2001", "-0o20001", "-0b10000000000001"),
        nc(-8192, "-8192", "-0x2000", "-0o20000", "-0b10000000000000"),
        nc(-8191, "-8191", "-0x1fff", "-0o17777", "-0b1111111111111"),
        nc(-4097, "-4097", "-0x1001", "-0o10001", "-0b1000000000001"),
        nc(-4096, "-4096", "-0x1000", "-0o10000", "-0b1000000000000"),
        nc(-4095, "-4095", "-0xfff", "-0o7777", "-0b111111111111"),
        nc(-2147, "-2147", "-0x863", "-0o4143", "-0b100001100011"),
        nc(-2049, "-2049", "-0x801", "-0o4001", "-0b100000000001"),
        nc(-2048, "-2048", "-0x800", "-0o4000", "-0b100000000000"),
        nc(-2047, "-2047", "-0x7ff", "-0o3777", "-0b11111111111"),
        nc(-1025, "-1025", "-0x401", "-0o2001", "-0b10000000001"),
        nc(-1024, "-1024", "-0x400", "-0o2000", "-0b10000000000"),
        nc(-1023, "-1023", "-0x3ff", "-0o1777", "-0b1111111111"),
        nc(-1001, "-1001", "-0x3e9", "-0o1751", "-0b1111101001"),
        nc(-1000, "-1000", "-0x3e8", "-0o1750", "-0b1111101000"),
        nc(-999, "-999", "-0x3e7", "-0o1747", "-0b1111100111"),
        nc(-513, "-513", "-0x201", "-0o1001", "-0b1000000001"),
        nc(-512, "-512", "-0x200", "-0o1000", "-0b1000000000"),
        nc(-511, "-511", "-0x1ff", "-0o777", "-0b111111111"),
        nc(-257, "-257", "-0x101", "-0o401", "-0b100000001"),
        nc(-256, "-256", "-0x100", "-0o400", "-0b100000000"),
        nc(-255, "-255", "-0xff", "-0o377", "-0b11111111"),
        nc(-214, "-214", "-0xd6", "-0o326", "-0b11010110"),
        nc(-129, "-129", "-0x81", "-0o201", "-0b10000001"),
        nc(-128, "-128", "-0x80", "-0o200", "-0b10000000"),
        nc(-127, "-127", "-0x7f", "-0o177", "-0b1111111"),
        nc(-101, "-101", "-0x65", "-0o145", "-0b1100101"),
        nc(-100, "-100", "-0x64", "-0o144", "-0b1100100"),
        nc(-99, "-99", "-0x63", "-0o143", "-0b1100011"),
        nc(-65, "-65", "-0x41", "-0o101", "-0b1000001"),
        nc(-64, "-64", "-0x40", "-0o100", "-0b1000000"),
        nc(-63, "-63", "-0x3f", "-0o77", "-0b111111"),
        nc(-33, "-33", "-0x21", "-0o41", "-0b100001"),
        nc(-32, "-32", "-0x20", "-0o40", "-0b100000"),
        nc(-31, "-31", "-0x1f", "-0o37", "-0b11111"),
        nc(-21, "-21", "-0x15", "-0o25", "-0b10101"),
        nc(-17, "-17", "-0x11", "-0o21", "-0b10001"),
        nc(-16, "-16", "-0x10", "-0o20", "-0b10000"),
        nc(-15, "-15", "-0xf", "-0o17", "-0b1111"),
        nc(-11, "-11", "-0xb", "-0o13", "-0b1011"),
        nc(-10, "-10", "-0xa", "-0o12", "-0b1010"),
        nc(-9, "-9", "-0x9", "-0o11", "-0b1001"),
        nc(-8, "-8", "-0x8", "-0o10", "-0b1000"),
        nc(-7, "-7", "-0x7", "-0o7", "-0b111"),
        nc(-6, "-6", "-0x6", "-0o6", "-0b110"),
        nc(-5, "-5", "-0x5", "-0o5", "-0b101"),
        nc(-4, "-4", "-0x4", "-0o4", "-0b100"),
        nc(-3, "-3", "-0x3", "-0o3", "-0b11"),
        nc(-2, "-2", "-0x2", "-0o2", "-0b10"),
        nc(-1, "-1", "-0x1", "-0o1", "-0b1"),
        nc(0, "0", "0x0", "0o0", "0b0"),
        nc(1, "1", "0x1", "0o1", "0b1"),
        nc(2, "2", "0x2", "0o2", "0b10"),
        nc(3, "3", "0x3", "0o3", "0b11"),
        nc(4, "4", "0x4", "0o4", "0b100"),
        nc(5, "5", "0x5", "0o5", "0b101"),
        nc(6, "6", "0x6", "0o6", "0b110"),
        nc(7, "7", "0x7", "0o7", "0b111"),
        nc(8, "8", "0x8", "0o10", "0b1000"),
        nc(9, "9", "0x9", "0o11", "0b1001"),
        nc(10, "10", "0xa", "0o12", "0b1010"),
        nc(11, "11", "0xb", "0o13", "0b1011"),
        nc(15, "15", "0xf", "0o17", "0b1111"),
        nc(16, "16", "0x10", "0o20", "0b10000"),
        nc(17, "17", "0x11", "0o21", "0b10001"),
        nc(20, "20", "0x14", "0o24", "0b10100"),
        nc(21, "21", "0x15", "0o25", "0b10101"),
        nc(22, "22", "0x16", "0o26", "0b10110"),
        nc(31, "31", "0x1f", "0o37", "0b11111"),
        nc(32, "32", "0x20", "0o40", "0b100000"),
        nc(33, "33", "0x21", "0o41", "0b100001"),
        nc(63, "63", "0x3f", "0o77", "0b111111"),
        nc(64, "64", "0x40", "0o100", "0b1000000"),
        nc(65, "65", "0x41", "0o101", "0b1000001"),
        nc(99, "99", "0x63", "0o143", "0b1100011"),
        nc(100, "100", "0x64", "0o144", "0b1100100"),
        nc(101, "101", "0x65", "0o145", "0b1100101"),
        nc(127, "127", "0x7f", "0o177", "0b1111111"),
        nc(128, "128", "0x80", "0o200", "0b10000000"),
        nc(129, "129", "0x81", "0o201", "0b10000001"),
        nc(213, "213", "0xd5", "0o325", "0b11010101"),
        nc(214, "214", "0xd6", "0o326", "0b11010110"),
        nc(215, "215", "0xd7", "0o327", "0b11010111"),
        nc(255, "255", "0xff", "0o377", "0b11111111"),
        nc(256, "256", "0x100", "0o400", "0b100000000"),
        nc(257, "257", "0x101", "0o401", "0b100000001"),
        nc(511, "511", "0x1ff", "0o777", "0b111111111"),
        nc(512, "512", "0x200", "0o1000", "0b1000000000"),
        nc(513, "513", "0x201", "0o1001", "0b1000000001"),
        nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
        nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
        nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
        nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
        nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
        nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
        nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
        nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
        nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
        nc(2146, "2146", "0x862", "0o4142", "0b100001100010"),
        nc(2147, "2147", "0x863", "0o4143", "0b100001100011"),
        nc(2148, "2148", "0x864", "0o4144", "0b100001100100"),
        nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
        nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
        nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
        nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
        nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
        nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
        nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
        nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
        nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
        nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
        nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
        nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
        nc(21473, "21473", "0x53e1", "0o51741", "0b101001111100001"),
        nc(21474, "21474", "0x53e2", "0o51742", "0b101001111100010"),
        nc(21475, "21475", "0x53e3", "0o51743", "0b101001111100011"),
        nc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111"),
        nc(32768, "32768", "0x8000", "0o100000", "0b1000000000000000"),
        nc(32769, "32769", "0x8001", "0o100001", "0b1000000000000001"),
        nc(65535, "65535", "0xffff", "0o177777", "0b1111111111111111"),
        nc(65536, "65536", "0x10000", "0o200000", "0b10000000000000000"),
        nc(65537, "65537", "0x10001", "0o200001", "0b10000000000000001"),
        nc(99999, "99999", "0x1869f", "0o303237", "0b11000011010011111"),
        nc(100000, "100000", "0x186a0", "0o303240", "0b11000011010100000"),
        nc(100001, "100001", "0x186a1", "0o303241", "0b11000011010100001"),
        nc(131071, "131071", "0x1ffff", "0o377777", "0b11111111111111111"),
        nc(131072, "131072", "0x20000", "0o400000", "0b100000000000000000"),
        nc(131073, "131073", "0x20001", "0o400001", "0b100000000000000001"),
        nc(214747, "214747", "0x346db", "0o643333", "0b110100011011011011"),
        nc(214748, "214748", "0x346dc", "0o643334", "0b110100011011011100"),
        nc(214749, "214749", "0x346dd", "0o643335", "0b110100011011011101"),
        nc(262143, "262143", "0x3ffff", "0o777777", "0b111111111111111111"),
        nc(262144, "262144", "0x40000", "0o1000000", "0b1000000000000000000"),
        nc(262145, "262145", "0x40001", "0o1000001", "0b1000000000000000001"),
        nc(524287, "524287", "0x7ffff", "0o1777777", "0b1111111111111111111"),
        nc(524288, "524288", "0x80000", "0o2000000", "0b10000000000000000000"),
        nc(524289, "524289", "0x80001", "0o2000001", "0b10000000000000000001"),
        nc(999999, "999999", "0xf423f", "0o3641077", "0b11110100001000111111"),
        nc(1000000, "1000000", "0xf4240", "0o3641100", "0b11110100001001000000"),
        nc(1000001, "1000001", "0xf4241", "0o3641101", "0b11110100001001000001"),
        nc(1048575, "1048575", "0xfffff", "0o3777777", "0b11111111111111111111"),
        nc(1048576, "1048576", "0x100000", "0o4000000", "0b100000000000000000000"),
        nc(1048577, "1048577", "0x100001", "0o4000001", "0b100000000000000000001"),
        nc(2097151, "2097151", "0x1fffff", "0o7777777", "0b111111111111111111111"),
        nc(2097152, "2097152", "0x200000", "0o10000000", "0b1000000000000000000000"),
        nc(2097153, "2097153", "0x200001", "0o10000001", "0b1000000000000000000001"),
        nc(2147482, "2147482", "0x20c49a", "0o10142232", "0b1000001100010010011010"),
        nc(2147483, "2147483", "0x20c49b", "0o10142233", "0b1000001100010010011011"),
        nc(2147484, "2147484", "0x20c49c", "0o10142234", "0b1000001100010010011100"),
        nc(4194303, "4194303", "0x3fffff", "0o17777777", "0b1111111111111111111111"),
        nc(4194304, "4194304", "0x400000", "0o20000000", "0b10000000000000000000000"),
        nc(4194305, "4194305", "0x400001", "0o20000001", "0b10000000000000000000001"),
        nc(8388607, "8388607", "0x7fffff", "0o37777777", "0b11111111111111111111111"),
        nc(8388608, "8388608", "0x800000", "0o40000000", "0b100000000000000000000000"),
        nc(8388609, "8388609", "0x800001", "0o40000001", "0b100000000000000000000001"),
        nc(9999999, "9999999", "0x98967f", "0o46113177", "0b100110001001011001111111"),
        nc(10000000, "10000000", "0x989680", "0o46113200", "0b100110001001011010000000"),
        nc(10000001, "10000001", "0x989681", "0o46113201", "0b100110001001011010000001"),
        nc(16777215, "16777215", "0xffffff", "0o77777777", "0b111111111111111111111111"),
        nc(16777216, "16777216", "0x1000000", "0o100000000", "0b1000000000000000000000000"),
        nc(16777217, "16777217", "0x1000001", "0o100000001", "0b1000000000000000000000001"),
        nc(21474835, "21474835", "0x147ae13", "0o121727023", "0b1010001111010111000010011"),
        nc(21474836, "21474836", "0x147ae14", "0o121727024", "0b1010001111010111000010100"),
        nc(21474837, "21474837", "0x147ae15", "0o121727025", "0b1010001111010111000010101"),
        nc(33554431, "33554431", "0x1ffffff", "0o177777777", "0b1111111111111111111111111"),
        nc(33554432, "33554432", "0x2000000", "0o200000000", "0b10000000000000000000000000"),
        nc(33554433, "33554433", "0x2000001", "0o200000001", "0b10000000000000000000000001"),
        nc(67108863, "67108863", "0x3ffffff", "0o377777777", "0b11111111111111111111111111"),
        nc(67108864, "67108864", "0x4000000", "0o400000000", "0b100000000000000000000000000"),
        nc(67108865, "67108865", "0x4000001", "0o400000001", "0b100000000000000000000000001"),
        nc(99999999, "99999999", "0x5f5e0ff", "0o575360377", "0b101111101011110000011111111"),
        nc(100000000, "100000000", "0x5f5e100", "0o575360400", "0b101111101011110000100000000"),
        nc(100000001, "100000001", "0x5f5e101", "0o575360401", "0b101111101011110000100000001"),
        nc(134217727, "134217727", "0x7ffffff", "0o777777777", "0b111111111111111111111111111"),
        nc(134217728, "134217728", "0x8000000", "0o1000000000", "0b1000000000000000000000000000"),
        nc(134217729, "134217729", "0x8000001", "0o1000000001", "0b1000000000000000000000000001"),
        nc(214748363, "214748363", "0xccccccb", "0o1463146313", "0b1100110011001100110011001011"),
        nc(214748364, "214748364", "0xccccccc", "0o1463146314", "0b1100110011001100110011001100"),
        nc(214748365, "214748365", "0xccccccd", "0o1463146315", "0b1100110011001100110011001101"),
        nc(268435455, "268435455", "0xfffffff", "0o1777777777", "0b1111111111111111111111111111"),
        nc(268435456, "268435456", "0x10000000", "0o2000000000", "0b10000000000000000000000000000"),
        nc(268435457, "268435457", "0x10000001", "0o2000000001", "0b10000000000000000000000000001"),
        nc(536870911, "536870911", "0x1fffffff", "0o3777777777", "0b11111111111111111111111111111"),
        nc(536870912, "536870912", "0x20000000", "0o4000000000", "0b100000000000000000000000000000"),
        nc(536870913, "536870913", "0x20000001", "0o4000000001", "0b100000000000000000000000000001"),
        nc(999999999, "999999999", "0x3b9ac9ff", "0o7346544777", "0b111011100110101100100111111111"),
        nc(1000000000, "1000000000", "0x3b9aca00", "0o7346545000", "0b111011100110101100101000000000"),
        nc(1000000001, "1000000001", "0x3b9aca01", "0o7346545001", "0b111011100110101100101000000001"),
        nc(1073741823, "1073741823", "0x3fffffff", "0o7777777777", "0b111111111111111111111111111111"),
        nc(1073741824, "1073741824", "0x40000000", "0o10000000000", "0b1000000000000000000000000000000"),
        nc(1073741825, "1073741825", "0x40000001", "0o10000000001", "0b1000000000000000000000000000001"),
        nc(2147483642, "2147483642", "0x7ffffffa", "0o17777777772", "0b1111111111111111111111111111010"),
        nc(2147483643, "2147483643", "0x7ffffffb", "0o17777777773", "0b1111111111111111111111111111011"),
        nc(2147483644, "2147483644", "0x7ffffffc", "0o17777777774", "0b1111111111111111111111111111100"),
        nc(2147483645, "2147483645", "0x7ffffffd", "0o17777777775", "0b1111111111111111111111111111101"),
        pc(2147483646, "2147483646", "0x7ffffffe", "0o17777777776", "0b1111111111111111111111111111110", 2147483647),
        pc(2147483647, "2147483647", "0x7fffffff", "0o17777777777", "0b1111111111111111111111111111111", -2147483648),
        #undef nc
        #undef pc
    };
};


template<>
struct numbers<uint32_t>
{
    using value_type = uint32_t;
    static C4_INLINE_CONSTEXPR const number_case<uint32_t> vals[] = {
        #define nc(val, dec, hex, bin, oct) nc_(UINT32_C, val, dec, hex, bin, oct)
        #define pc(val, dec, hex, bin, oct, valp1) ncp_(UINT32_C, val, dec, hex, bin, oct, valp1)
        pc(0, "0", "0x0", "0o0", "0b0", 1),
        nc(1, "1", "0x1", "0o1", "0b1"),
        nc(2, "2", "0x2", "0o2", "0b10"),
        nc(3, "3", "0x3", "0o3", "0b11"),
        nc(4, "4", "0x4", "0o4", "0b100"),
        nc(5, "5", "0x5", "0o5", "0b101"),
        nc(6, "6", "0x6", "0o6", "0b110"),
        nc(7, "7", "0x7", "0o7", "0b111"),
        nc(8, "8", "0x8", "0o10", "0b1000"),
        nc(9, "9", "0x9", "0o11", "0b1001"),
        nc(10, "10", "0xa", "0o12", "0b1010"),
        nc(11, "11", "0xb", "0o13", "0b1011"),
        nc(15, "15", "0xf", "0o17", "0b1111"),
        nc(16, "16", "0x10", "0o20", "0b10000"),
        nc(17, "17", "0x11", "0o21", "0b10001"),
        nc(31, "31", "0x1f", "0o37", "0b11111"),
        nc(32, "32", "0x20", "0o40", "0b100000"),
        nc(33, "33", "0x21", "0o41", "0b100001"),
        nc(41, "41", "0x29", "0o51", "0b101001"),
        nc(42, "42", "0x2a", "0o52", "0b101010"),
        nc(43, "43", "0x2b", "0o53", "0b101011"),
        nc(63, "63", "0x3f", "0o77", "0b111111"),
        nc(64, "64", "0x40", "0o100", "0b1000000"),
        nc(65, "65", "0x41", "0o101", "0b1000001"),
        nc(99, "99", "0x63", "0o143", "0b1100011"),
        nc(100, "100", "0x64", "0o144", "0b1100100"),
        nc(101, "101", "0x65", "0o145", "0b1100101"),
        nc(127, "127", "0x7f", "0o177", "0b1111111"),
        nc(128, "128", "0x80", "0o200", "0b10000000"),
        nc(129, "129", "0x81", "0o201", "0b10000001"),
        nc(255, "255", "0xff", "0o377", "0b11111111"),
        nc(256, "256", "0x100", "0o400", "0b100000000"),
        nc(257, "257", "0x101", "0o401", "0b100000001"),
        nc(428, "428", "0x1ac", "0o654", "0b110101100"),
        nc(429, "429", "0x1ad", "0o655", "0b110101101"),
        nc(430, "430", "0x1ae", "0o656", "0b110101110"),
        nc(511, "511", "0x1ff", "0o777", "0b111111111"),
        nc(512, "512", "0x200", "0o1000", "0b1000000000"),
        nc(513, "513", "0x201", "0o1001", "0b1000000001"),
        nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
        nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
        nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
        nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
        nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
        nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
        nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
        nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
        nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
        nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
        nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
        nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
        nc(4293, "4293", "0x10c5", "0o10305", "0b1000011000101"),
        nc(4294, "4294", "0x10c6", "0o10306", "0b1000011000110"),
        nc(4295, "4295", "0x10c7", "0o10307", "0b1000011000111"),
        nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
        nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
        nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
        nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
        nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
        nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
        nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
        nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
        nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
        nc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111"),
        nc(32768, "32768", "0x8000", "0o100000", "0b1000000000000000"),
        nc(32769, "32769", "0x8001", "0o100001", "0b1000000000000001"),
        nc(42948, "42948", "0xa7c4", "0o123704", "0b1010011111000100"),
        nc(42949, "42949", "0xa7c5", "0o123705", "0b1010011111000101"),
        nc(42950, "42950", "0xa7c6", "0o123706", "0b1010011111000110"),
        nc(65535, "65535", "0xffff", "0o177777", "0b1111111111111111"),
        nc(65536, "65536", "0x10000", "0o200000", "0b10000000000000000"),
        nc(65537, "65537", "0x10001", "0o200001", "0b10000000000000001"),
        nc(99999, "99999", "0x1869f", "0o303237", "0b11000011010011111"),
        nc(100000, "100000", "0x186a0", "0o303240", "0b11000011010100000"),
        nc(100001, "100001", "0x186a1", "0o303241", "0b11000011010100001"),
        nc(131071, "131071", "0x1ffff", "0o377777", "0b11111111111111111"),
        nc(131072, "131072", "0x20000", "0o400000", "0b100000000000000000"),
        nc(131073, "131073", "0x20001", "0o400001", "0b100000000000000001"),
        nc(262143, "262143", "0x3ffff", "0o777777", "0b111111111111111111"),
        nc(262144, "262144", "0x40000", "0o1000000", "0b1000000000000000000"),
        nc(262145, "262145", "0x40001", "0o1000001", "0b1000000000000000001"),
        nc(429495, "429495", "0x68db7", "0o1506667", "0b1101000110110110111"),
        nc(429496, "429496", "0x68db8", "0o1506670", "0b1101000110110111000"),
        nc(429497, "429497", "0x68db9", "0o1506671", "0b1101000110110111001"),
        nc(524287, "524287", "0x7ffff", "0o1777777", "0b1111111111111111111"),
        nc(524288, "524288", "0x80000", "0o2000000", "0b10000000000000000000"),
        nc(524289, "524289", "0x80001", "0o2000001", "0b10000000000000000001"),
        nc(999999, "999999", "0xf423f", "0o3641077", "0b11110100001000111111"),
        nc(1000000, "1000000", "0xf4240", "0o3641100", "0b11110100001001000000"),
        nc(1000001, "1000001", "0xf4241", "0o3641101", "0b11110100001001000001"),
        nc(1048575, "1048575", "0xfffff", "0o3777777", "0b11111111111111111111"),
        nc(1048576, "1048576", "0x100000", "0o4000000", "0b100000000000000000000"),
        nc(1048577, "1048577", "0x100001", "0o4000001", "0b100000000000000000001"),
        nc(2097151, "2097151", "0x1fffff", "0o7777777", "0b111111111111111111111"),
        nc(2097152, "2097152", "0x200000", "0o10000000", "0b1000000000000000000000"),
        nc(2097153, "2097153", "0x200001", "0o10000001", "0b1000000000000000000001"),
        nc(4194303, "4194303", "0x3fffff", "0o17777777", "0b1111111111111111111111"),
        nc(4194304, "4194304", "0x400000", "0o20000000", "0b10000000000000000000000"),
        nc(4194305, "4194305", "0x400001", "0o20000001", "0b10000000000000000000001"),
        nc(4294966, "4294966", "0x418936", "0o20304466", "0b10000011000100100110110"),
        nc(4294967, "4294967", "0x418937", "0o20304467", "0b10000011000100100110111"),
        nc(4294968, "4294968", "0x418938", "0o20304470", "0b10000011000100100111000"),
        nc(8388607, "8388607", "0x7fffff", "0o37777777", "0b11111111111111111111111"),
        nc(8388608, "8388608", "0x800000", "0o40000000", "0b100000000000000000000000"),
        nc(8388609, "8388609", "0x800001", "0o40000001", "0b100000000000000000000001"),
        nc(9999999, "9999999", "0x98967f", "0o46113177", "0b100110001001011001111111"),
        nc(10000000, "10000000", "0x989680", "0o46113200", "0b100110001001011010000000"),
        nc(10000001, "10000001", "0x989681", "0o46113201", "0b100110001001011010000001"),
        nc(16777215, "16777215", "0xffffff", "0o77777777", "0b111111111111111111111111"),
        nc(16777216, "16777216", "0x1000000", "0o100000000", "0b1000000000000000000000000"),
        nc(16777217, "16777217", "0x1000001", "0o100000001", "0b1000000000000000000000001"),
        nc(33554431, "33554431", "0x1ffffff", "0o177777777", "0b1111111111111111111111111"),
        nc(33554432, "33554432", "0x2000000", "0o200000000", "0b10000000000000000000000000"),
        nc(33554433, "33554433", "0x2000001", "0o200000001", "0b10000000000000000000000001"),
        nc(42949671, "42949671", "0x28f5c27", "0o243656047", "0b10100011110101110000100111"),
        nc(42949672, "42949672", "0x28f5c28", "0o243656050", "0b10100011110101110000101000"),
        nc(42949673, "42949673", "0x28f5c29", "0o243656051", "0b10100011110101110000101001"),
        nc(67108863, "67108863", "0x3ffffff", "0o377777777", "0b11111111111111111111111111"),
        nc(67108864, "67108864", "0x4000000", "0o400000000", "0b100000000000000000000000000"),
        nc(67108865, "67108865", "0x4000001", "0o400000001", "0b100000000000000000000000001"),
        nc(99999999, "99999999", "0x5f5e0ff", "0o575360377", "0b101111101011110000011111111"),
        nc(100000000, "100000000", "0x5f5e100", "0o575360400", "0b101111101011110000100000000"),
        nc(100000001, "100000001", "0x5f5e101", "0o575360401", "0b101111101011110000100000001"),
        nc(134217727, "134217727", "0x7ffffff", "0o777777777", "0b111111111111111111111111111"),
        nc(134217728, "134217728", "0x8000000", "0o1000000000", "0b1000000000000000000000000000"),
        nc(134217729, "134217729", "0x8000001", "0o1000000001", "0b1000000000000000000000000001"),
        nc(268435455, "268435455", "0xfffffff", "0o1777777777", "0b1111111111111111111111111111"),
        nc(268435456, "268435456", "0x10000000", "0o2000000000", "0b10000000000000000000000000000"),
        nc(268435457, "268435457", "0x10000001", "0o2000000001", "0b10000000000000000000000000001"),
        nc(429496728, "429496728", "0x19999998", "0o3146314630", "0b11001100110011001100110011000"),
        nc(429496729, "429496729", "0x19999999", "0o3146314631", "0b11001100110011001100110011001"),
        nc(429496730, "429496730", "0x1999999a", "0o3146314632", "0b11001100110011001100110011010"),
        nc(536870911, "536870911", "0x1fffffff", "0o3777777777", "0b11111111111111111111111111111"),
        nc(536870912, "536870912", "0x20000000", "0o4000000000", "0b100000000000000000000000000000"),
        nc(536870913, "536870913", "0x20000001", "0o4000000001", "0b100000000000000000000000000001"),
        nc(999999999, "999999999", "0x3b9ac9ff", "0o7346544777", "0b111011100110101100100111111111"),
        nc(1000000000, "1000000000", "0x3b9aca00", "0o7346545000", "0b111011100110101100101000000000"),
        nc(1000000001, "1000000001", "0x3b9aca01", "0o7346545001", "0b111011100110101100101000000001"),
        nc(1073741823, "1073741823", "0x3fffffff", "0o7777777777", "0b111111111111111111111111111111"),
        nc(1073741824, "1073741824", "0x40000000", "0o10000000000", "0b1000000000000000000000000000000"),
        nc(1073741825, "1073741825", "0x40000001", "0o10000000001", "0b1000000000000000000000000000001"),
        nc(2147483647, "2147483647", "0x7fffffff", "0o17777777777", "0b1111111111111111111111111111111"),
        nc(2147483648, "2147483648", "0x80000000", "0o20000000000", "0b10000000000000000000000000000000"),
        nc(2147483649, "2147483649", "0x80000001", "0o20000000001", "0b10000000000000000000000000000001"),
        pc(4294967291, "4294967291", "0xfffffffb", "0o37777777773", "0b11111111111111111111111111111011", 4294967292),
        pc(4294967292, "4294967292", "0xfffffffc", "0o37777777774", "0b11111111111111111111111111111100", 4294967293),
        pc(4294967293, "4294967293", "0xfffffffd", "0o37777777775", "0b11111111111111111111111111111101", 4294967294),
        pc(4294967294, "4294967294", "0xfffffffe", "0o37777777776", "0b11111111111111111111111111111110", 4294967295),
        pc(4294967295, "4294967295", "0xffffffff", "0o37777777777", "0b11111111111111111111111111111111", 0),
#undef nc
#undef pc
    };
};


template<>
struct numbers<int64_t>
{
    using value_type = int64_t;
    static C4_INLINE_CONSTEXPR const number_case<int64_t> vals[] = {
        #define nc(val, dec, hex, bin, oct) nc_(INT64_C, val, dec, hex, bin, oct)
        #define ncm1(val, dec, hex, bin, oct) nc_(INT64_1, val, dec, hex, bin, oct)
        #define pc(val, dec, hex, bin, oct, valp1) ncp_(INT64_C, val, dec, hex, bin, oct, valp1)
        #define pcm1(val, dec, hex, bin, oct, valp1) ncp__(INT64_C, val, dec, hex, bin, oct, INT64_1, valp1)
        ncm1(-9223372036854775807, "-9223372036854775808", "-0x8000000000000000", "-0o1000000000000000000000", "-0b1000000000000000000000000000000000000000000000000000000000000000"),
        nc(-9223372036854775807, "-9223372036854775807", "-0x7fffffffffffffff", "-0o777777777777777777777", "-0b111111111111111111111111111111111111111111111111111111111111111"),
        nc(-9223372036854775806, "-9223372036854775806", "-0x7ffffffffffffffe", "-0o777777777777777777776", "-0b111111111111111111111111111111111111111111111111111111111111110"),
        nc(-9223372036854775805, "-9223372036854775805", "-0x7ffffffffffffffd", "-0o777777777777777777775", "-0b111111111111111111111111111111111111111111111111111111111111101"),
        nc(-9223372036854775804, "-9223372036854775804", "-0x7ffffffffffffffc", "-0o777777777777777777774", "-0b111111111111111111111111111111111111111111111111111111111111100"),
        nc(-9223372036854775803, "-9223372036854775803", "-0x7ffffffffffffffb", "-0o777777777777777777773", "-0b111111111111111111111111111111111111111111111111111111111111011"),
        nc(-4611686018427387905, "-4611686018427387905", "-0x4000000000000001", "-0o400000000000000000001", "-0b100000000000000000000000000000000000000000000000000000000000001"),
        nc(-4611686018427387904, "-4611686018427387904", "-0x4000000000000000", "-0o400000000000000000000", "-0b100000000000000000000000000000000000000000000000000000000000000"),
        nc(-4611686018427387903, "-4611686018427387903", "-0x3fffffffffffffff", "-0o377777777777777777777", "-0b11111111111111111111111111111111111111111111111111111111111111"),
        nc(-2305843009213693953, "-2305843009213693953", "-0x2000000000000001", "-0o200000000000000000001", "-0b10000000000000000000000000000000000000000000000000000000000001"),
        nc(-2305843009213693952, "-2305843009213693952", "-0x2000000000000000", "-0o200000000000000000000", "-0b10000000000000000000000000000000000000000000000000000000000000"),
        nc(-2305843009213693951, "-2305843009213693951", "-0x1fffffffffffffff", "-0o177777777777777777777", "-0b1111111111111111111111111111111111111111111111111111111111111"),
        nc(-1152921504606846977, "-1152921504606846977", "-0x1000000000000001", "-0o100000000000000000001", "-0b1000000000000000000000000000000000000000000000000000000000001"),
        nc(-1152921504606846976, "-1152921504606846976", "-0x1000000000000000", "-0o100000000000000000000", "-0b1000000000000000000000000000000000000000000000000000000000000"),
        nc(-1152921504606846975, "-1152921504606846975", "-0xfffffffffffffff", "-0o77777777777777777777", "-0b111111111111111111111111111111111111111111111111111111111111"),
        nc(-1000000000000000001, "-1000000000000000001", "-0xde0b6b3a7640001", "-0o67405553164731000001", "-0b110111100000101101101011001110100111011001000000000000000001"),
        nc(-1000000000000000000, "-1000000000000000000", "-0xde0b6b3a7640000", "-0o67405553164731000000", "-0b110111100000101101101011001110100111011001000000000000000000"),
        nc(-999999999999999999, "-999999999999999999", "-0xde0b6b3a763ffff", "-0o67405553164730777777", "-0b110111100000101101101011001110100111011000111111111111111111"),
        nc(-922337203685477632, "-922337203685477632", "-0xccccccccccccd00", "-0o63146314631463146400", "-0b110011001100110011001100110011001100110011001100110100000000"),
        nc(-576460752303423489, "-576460752303423489", "-0x800000000000001", "-0o40000000000000000001", "-0b100000000000000000000000000000000000000000000000000000000001"),
        nc(-576460752303423488, "-576460752303423488", "-0x800000000000000", "-0o40000000000000000000", "-0b100000000000000000000000000000000000000000000000000000000000"),
        nc(-576460752303423487, "-576460752303423487", "-0x7ffffffffffffff", "-0o37777777777777777777", "-0b11111111111111111111111111111111111111111111111111111111111"),
        nc(-288230376151711745, "-288230376151711745", "-0x400000000000001", "-0o20000000000000000001", "-0b10000000000000000000000000000000000000000000000000000000001"),
        nc(-288230376151711744, "-288230376151711744", "-0x400000000000000", "-0o20000000000000000000", "-0b10000000000000000000000000000000000000000000000000000000000"),
        nc(-288230376151711743, "-288230376151711743", "-0x3ffffffffffffff", "-0o17777777777777777777", "-0b1111111111111111111111111111111111111111111111111111111111"),
        nc(-144115188075855873, "-144115188075855873", "-0x200000000000001", "-0o10000000000000000001", "-0b1000000000000000000000000000000000000000000000000000000001"),
        nc(-144115188075855872, "-144115188075855872", "-0x200000000000000", "-0o10000000000000000000", "-0b1000000000000000000000000000000000000000000000000000000000"),
        nc(-144115188075855871, "-144115188075855871", "-0x1ffffffffffffff", "-0o7777777777777777777", "-0b111111111111111111111111111111111111111111111111111111111"),
        nc(-100000000000000001, "-100000000000000001", "-0x16345785d8a0001", "-0o5432127413542400001", "-0b101100011010001010111100001011101100010100000000000000001"),
        nc(-100000000000000000, "-100000000000000000", "-0x16345785d8a0000", "-0o5432127413542400000", "-0b101100011010001010111100001011101100010100000000000000000"),
        nc(-99999999999999999, "-99999999999999999", "-0x16345785d89ffff", "-0o5432127413542377777", "-0b101100011010001010111100001011101100010011111111111111111"),
        nc(-92233720368547760, "-92233720368547760", "-0x147ae147ae147b0", "-0o5075341217270243660", "-0b101000111101011100001010001111010111000010100011110110000"),
        nc(-72057594037927937, "-72057594037927937", "-0x100000000000001", "-0o4000000000000000001", "-0b100000000000000000000000000000000000000000000000000000001"),
        nc(-72057594037927936, "-72057594037927936", "-0x100000000000000", "-0o4000000000000000000", "-0b100000000000000000000000000000000000000000000000000000000"),
        nc(-72057594037927935, "-72057594037927935", "-0xffffffffffffff", "-0o3777777777777777777", "-0b11111111111111111111111111111111111111111111111111111111"),
        nc(-36028797018963969, "-36028797018963969", "-0x80000000000001", "-0o2000000000000000001", "-0b10000000000000000000000000000000000000000000000000000001"),
        nc(-36028797018963968, "-36028797018963968", "-0x80000000000000", "-0o2000000000000000000", "-0b10000000000000000000000000000000000000000000000000000000"),
        nc(-36028797018963967, "-36028797018963967", "-0x7fffffffffffff", "-0o1777777777777777777", "-0b1111111111111111111111111111111111111111111111111111111"),
        nc(-18014398509481985, "-18014398509481985", "-0x40000000000001", "-0o1000000000000000001", "-0b1000000000000000000000000000000000000000000000000000001"),
        nc(-18014398509481984, "-18014398509481984", "-0x40000000000000", "-0o1000000000000000000", "-0b1000000000000000000000000000000000000000000000000000000"),
        nc(-18014398509481983, "-18014398509481983", "-0x3fffffffffffff", "-0o777777777777777777", "-0b111111111111111111111111111111111111111111111111111111"),
        nc(-10000000000000001, "-10000000000000001", "-0x2386f26fc10001", "-0o434157115760200001", "-0b100011100001101111001001101111110000010000000000000001"),
        nc(-10000000000000000, "-10000000000000000", "-0x2386f26fc10000", "-0o434157115760200000", "-0b100011100001101111001001101111110000010000000000000000"),
        nc(-9999999999999999, "-9999999999999999", "-0x2386f26fc0ffff", "-0o434157115760177777", "-0b100011100001101111001001101111110000001111111111111111"),
        nc(-9223372036854776, "-9223372036854776", "-0x20c49ba5e353f8", "-0o406111564570651770", "-0b100000110001001001101110100101111000110101001111111000"),
        nc(-9007199254740993, "-9007199254740993", "-0x20000000000001", "-0o400000000000000001", "-0b100000000000000000000000000000000000000000000000000001"),
        nc(-9007199254740992, "-9007199254740992", "-0x20000000000000", "-0o400000000000000000", "-0b100000000000000000000000000000000000000000000000000000"),
        nc(-9007199254740991, "-9007199254740991", "-0x1fffffffffffff", "-0o377777777777777777", "-0b11111111111111111111111111111111111111111111111111111"),
        nc(-4503599627370497, "-4503599627370497", "-0x10000000000001", "-0o200000000000000001", "-0b10000000000000000000000000000000000000000000000000001"),
        nc(-4503599627370496, "-4503599627370496", "-0x10000000000000", "-0o200000000000000000", "-0b10000000000000000000000000000000000000000000000000000"),
        nc(-4503599627370495, "-4503599627370495", "-0xfffffffffffff", "-0o177777777777777777", "-0b1111111111111111111111111111111111111111111111111111"),
        nc(-2251799813685249, "-2251799813685249", "-0x8000000000001", "-0o100000000000000001", "-0b1000000000000000000000000000000000000000000000000001"),
        nc(-2251799813685248, "-2251799813685248", "-0x8000000000000", "-0o100000000000000000", "-0b1000000000000000000000000000000000000000000000000000"),
        nc(-2251799813685247, "-2251799813685247", "-0x7ffffffffffff", "-0o77777777777777777", "-0b111111111111111111111111111111111111111111111111111"),
        nc(-1125899906842625, "-1125899906842625", "-0x4000000000001", "-0o40000000000000001", "-0b100000000000000000000000000000000000000000000000001"),
        nc(-1125899906842624, "-1125899906842624", "-0x4000000000000", "-0o40000000000000000", "-0b100000000000000000000000000000000000000000000000000"),
        nc(-1125899906842623, "-1125899906842623", "-0x3ffffffffffff", "-0o37777777777777777", "-0b11111111111111111111111111111111111111111111111111"),
        nc(-1000000000000001, "-1000000000000001", "-0x38d7ea4c68001", "-0o34327724461500001", "-0b11100011010111111010100100110001101000000000000001"),
        nc(-1000000000000000, "-1000000000000000", "-0x38d7ea4c68000", "-0o34327724461500000", "-0b11100011010111111010100100110001101000000000000000"),
        nc(-999999999999999, "-999999999999999", "-0x38d7ea4c67fff", "-0o34327724461477777", "-0b11100011010111111010100100110001100111111111111111"),
        nc(-922337203685477, "-922337203685477", "-0x346dc5d638865", "-0o32155613530704145", "-0b11010001101101110001011101011000111000100001100101"),
        nc(-562949953421313, "-562949953421313", "-0x2000000000001", "-0o20000000000000001", "-0b10000000000000000000000000000000000000000000000001"),
        nc(-562949953421312, "-562949953421312", "-0x2000000000000", "-0o20000000000000000", "-0b10000000000000000000000000000000000000000000000000"),
        nc(-562949953421311, "-562949953421311", "-0x1ffffffffffff", "-0o17777777777777777", "-0b1111111111111111111111111111111111111111111111111"),
        nc(-281474976710657, "-281474976710657", "-0x1000000000001", "-0o10000000000000001", "-0b1000000000000000000000000000000000000000000000001"),
        nc(-281474976710656, "-281474976710656", "-0x1000000000000", "-0o10000000000000000", "-0b1000000000000000000000000000000000000000000000000"),
        nc(-281474976710655, "-281474976710655", "-0xffffffffffff", "-0o7777777777777777", "-0b111111111111111111111111111111111111111111111111"),
        nc(-140737488355329, "-140737488355329", "-0x800000000001", "-0o4000000000000001", "-0b100000000000000000000000000000000000000000000001"),
        nc(-140737488355328, "-140737488355328", "-0x800000000000", "-0o4000000000000000", "-0b100000000000000000000000000000000000000000000000"),
        nc(-140737488355327, "-140737488355327", "-0x7fffffffffff", "-0o3777777777777777", "-0b11111111111111111111111111111111111111111111111"),
        nc(-100000000000001, "-100000000000001", "-0x5af3107a4001", "-0o2657142036440001", "-0b10110101111001100010000011110100100000000000001"),
        nc(-100000000000000, "-100000000000000", "-0x5af3107a4000", "-0o2657142036440000", "-0b10110101111001100010000011110100100000000000000"),
        nc(-99999999999999, "-99999999999999", "-0x5af3107a3fff", "-0o2657142036437777", "-0b10110101111001100010000011110100011111111111111"),
        nc(-92233720368547, "-92233720368547", "-0x53e2d6238da3", "-0o2476132610706643", "-0b10100111110001011010110001000111000110110100011"),
        nc(-70368744177665, "-70368744177665", "-0x400000000001", "-0o2000000000000001", "-0b10000000000000000000000000000000000000000000001"),
        nc(-70368744177664, "-70368744177664", "-0x400000000000", "-0o2000000000000000", "-0b10000000000000000000000000000000000000000000000"),
        nc(-70368744177663, "-70368744177663", "-0x3fffffffffff", "-0o1777777777777777", "-0b1111111111111111111111111111111111111111111111"),
        nc(-35184372088833, "-35184372088833", "-0x200000000001", "-0o1000000000000001", "-0b1000000000000000000000000000000000000000000001"),
        nc(-35184372088832, "-35184372088832", "-0x200000000000", "-0o1000000000000000", "-0b1000000000000000000000000000000000000000000000"),
        nc(-35184372088831, "-35184372088831", "-0x1fffffffffff", "-0o777777777777777", "-0b111111111111111111111111111111111111111111111"),
        nc(-17592186044417, "-17592186044417", "-0x100000000001", "-0o400000000000001", "-0b100000000000000000000000000000000000000000001"),
        nc(-17592186044416, "-17592186044416", "-0x100000000000", "-0o400000000000000", "-0b100000000000000000000000000000000000000000000"),
        nc(-17592186044415, "-17592186044415", "-0xfffffffffff", "-0o377777777777777", "-0b11111111111111111111111111111111111111111111"),
        nc(-10000000000001, "-10000000000001", "-0x9184e72a001", "-0o221411634520001", "-0b10010001100001001110011100101010000000000001"),
        nc(-10000000000000, "-10000000000000", "-0x9184e72a000", "-0o221411634520000", "-0b10010001100001001110011100101010000000000000"),
        nc(-9999999999999, "-9999999999999", "-0x9184e729fff", "-0o221411634517777", "-0b10010001100001001110011100101001111111111111"),
        nc(-9223372036854, "-9223372036854", "-0x8637bd05af6", "-0o206157364055366", "-0b10000110001101111011110100000101101011110110"),
        nc(-8796093022209, "-8796093022209", "-0x80000000001", "-0o200000000000001", "-0b10000000000000000000000000000000000000000001"),
        nc(-8796093022208, "-8796093022208", "-0x80000000000", "-0o200000000000000", "-0b10000000000000000000000000000000000000000000"),
        nc(-8796093022207, "-8796093022207", "-0x7ffffffffff", "-0o177777777777777", "-0b1111111111111111111111111111111111111111111"),
        nc(-4398046511105, "-4398046511105", "-0x40000000001", "-0o100000000000001", "-0b1000000000000000000000000000000000000000001"),
        nc(-4398046511104, "-4398046511104", "-0x40000000000", "-0o100000000000000", "-0b1000000000000000000000000000000000000000000"),
        nc(-4398046511103, "-4398046511103", "-0x3ffffffffff", "-0o77777777777777", "-0b111111111111111111111111111111111111111111"),
        nc(-2199023255553, "-2199023255553", "-0x20000000001", "-0o40000000000001", "-0b100000000000000000000000000000000000000001"),
        nc(-2199023255552, "-2199023255552", "-0x20000000000", "-0o40000000000000", "-0b100000000000000000000000000000000000000000"),
        nc(-2199023255551, "-2199023255551", "-0x1ffffffffff", "-0o37777777777777", "-0b11111111111111111111111111111111111111111"),
        nc(-1099511627777, "-1099511627777", "-0x10000000001", "-0o20000000000001", "-0b10000000000000000000000000000000000000001"),
        nc(-1099511627776, "-1099511627776", "-0x10000000000", "-0o20000000000000", "-0b10000000000000000000000000000000000000000"),
        nc(-1099511627775, "-1099511627775", "-0xffffffffff", "-0o17777777777777", "-0b1111111111111111111111111111111111111111"),
        nc(-1000000000001, "-1000000000001", "-0xe8d4a51001", "-0o16432451210001", "-0b1110100011010100101001010001000000000001"),
        nc(-1000000000000, "-1000000000000", "-0xe8d4a51000", "-0o16432451210000", "-0b1110100011010100101001010001000000000000"),
        nc(-999999999999, "-999999999999", "-0xe8d4a50fff", "-0o16432451207777", "-0b1110100011010100101001010000111111111111"),
        nc(-922337203685, "-922337203685", "-0xd6bf94d5e5", "-0o15327745152745", "-0b1101011010111111100101001101010111100101"),
        nc(-549755813889, "-549755813889", "-0x8000000001", "-0o10000000000001", "-0b1000000000000000000000000000000000000001"),
        nc(-549755813888, "-549755813888", "-0x8000000000", "-0o10000000000000", "-0b1000000000000000000000000000000000000000"),
        nc(-549755813887, "-549755813887", "-0x7fffffffff", "-0o7777777777777", "-0b111111111111111111111111111111111111111"),
        nc(-274877906945, "-274877906945", "-0x4000000001", "-0o4000000000001", "-0b100000000000000000000000000000000000001"),
        nc(-274877906944, "-274877906944", "-0x4000000000", "-0o4000000000000", "-0b100000000000000000000000000000000000000"),
        nc(-274877906943, "-274877906943", "-0x3fffffffff", "-0o3777777777777", "-0b11111111111111111111111111111111111111"),
        nc(-137438953473, "-137438953473", "-0x2000000001", "-0o2000000000001", "-0b10000000000000000000000000000000000001"),
        nc(-137438953472, "-137438953472", "-0x2000000000", "-0o2000000000000", "-0b10000000000000000000000000000000000000"),
        nc(-137438953471, "-137438953471", "-0x1fffffffff", "-0o1777777777777", "-0b1111111111111111111111111111111111111"),
        nc(-100000000001, "-100000000001", "-0x174876e801", "-0o1351035564001", "-0b1011101001000011101101110100000000001"),
        nc(-100000000000, "-100000000000", "-0x174876e800", "-0o1351035564000", "-0b1011101001000011101101110100000000000"),
        nc(-99999999999, "-99999999999", "-0x174876e7ff", "-0o1351035563777", "-0b1011101001000011101101110011111111111"),
        nc(-92233720368, "-92233720368", "-0x15798ee230", "-0o1257143561060", "-0b1010101111001100011101110001000110000"),
        nc(-68719476737, "-68719476737", "-0x1000000001", "-0o1000000000001", "-0b1000000000000000000000000000000000001"),
        nc(-68719476736, "-68719476736", "-0x1000000000", "-0o1000000000000", "-0b1000000000000000000000000000000000000"),
        nc(-68719476735, "-68719476735", "-0xfffffffff", "-0o777777777777", "-0b111111111111111111111111111111111111"),
        nc(-34359738369, "-34359738369", "-0x800000001", "-0o400000000001", "-0b100000000000000000000000000000000001"),
        nc(-34359738368, "-34359738368", "-0x800000000", "-0o400000000000", "-0b100000000000000000000000000000000000"),
        nc(-34359738367, "-34359738367", "-0x7ffffffff", "-0o377777777777", "-0b11111111111111111111111111111111111"),
        nc(-17179869185, "-17179869185", "-0x400000001", "-0o200000000001", "-0b10000000000000000000000000000000001"),
        nc(-17179869184, "-17179869184", "-0x400000000", "-0o200000000000", "-0b10000000000000000000000000000000000"),
        nc(-17179869183, "-17179869183", "-0x3ffffffff", "-0o177777777777", "-0b1111111111111111111111111111111111"),
        nc(-10000000001, "-10000000001", "-0x2540be401", "-0o112402762001", "-0b1001010100000010111110010000000001"),
        nc(-10000000000, "-10000000000", "-0x2540be400", "-0o112402762000", "-0b1001010100000010111110010000000000"),
        nc(-9999999999, "-9999999999", "-0x2540be3ff", "-0o112402761777", "-0b1001010100000010111110001111111111"),
        nc(-9223372036, "-9223372036", "-0x225c17d04", "-0o104560276404", "-0b1000100101110000010111110100000100"),
        nc(-8589934593, "-8589934593", "-0x200000001", "-0o100000000001", "-0b1000000000000000000000000000000001"),
        nc(-8589934592, "-8589934592", "-0x200000000", "-0o100000000000", "-0b1000000000000000000000000000000000"),
        nc(-8589934591, "-8589934591", "-0x1ffffffff", "-0o77777777777", "-0b111111111111111111111111111111111"),
        nc(-4294967297, "-4294967297", "-0x100000001", "-0o40000000001", "-0b100000000000000000000000000000001"),
        nc(-4294967296, "-4294967296", "-0x100000000", "-0o40000000000", "-0b100000000000000000000000000000000"),
        nc(-4294967295, "-4294967295", "-0xffffffff", "-0o37777777777", "-0b11111111111111111111111111111111"),
        nc(-2147483649, "-2147483649", "-0x80000001", "-0o20000000001", "-0b10000000000000000000000000000001"),
        nc(-2147483648, "-2147483648", "-0x80000000", "-0o20000000000", "-0b10000000000000000000000000000000"),
        nc(-2147483647, "-2147483647", "-0x7fffffff", "-0o17777777777", "-0b1111111111111111111111111111111"),
        nc(-1073741825, "-1073741825", "-0x40000001", "-0o10000000001", "-0b1000000000000000000000000000001"),
        nc(-1073741824, "-1073741824", "-0x40000000", "-0o10000000000", "-0b1000000000000000000000000000000"),
        nc(-1073741823, "-1073741823", "-0x3fffffff", "-0o7777777777", "-0b111111111111111111111111111111"),
        nc(-1000000001, "-1000000001", "-0x3b9aca01", "-0o7346545001", "-0b111011100110101100101000000001"),
        nc(-1000000000, "-1000000000", "-0x3b9aca00", "-0o7346545000", "-0b111011100110101100101000000000"),
        nc(-999999999, "-999999999", "-0x3b9ac9ff", "-0o7346544777", "-0b111011100110101100100111111111"),
        nc(-922337203, "-922337203", "-0x36f9bfb3", "-0o6676337663", "-0b110110111110011011111110110011"),
        nc(-536870913, "-536870913", "-0x20000001", "-0o4000000001", "-0b100000000000000000000000000001"),
        nc(-536870912, "-536870912", "-0x20000000", "-0o4000000000", "-0b100000000000000000000000000000"),
        nc(-536870911, "-536870911", "-0x1fffffff", "-0o3777777777", "-0b11111111111111111111111111111"),
        nc(-268435457, "-268435457", "-0x10000001", "-0o2000000001", "-0b10000000000000000000000000001"),
        nc(-268435456, "-268435456", "-0x10000000", "-0o2000000000", "-0b10000000000000000000000000000"),
        nc(-268435455, "-268435455", "-0xfffffff", "-0o1777777777", "-0b1111111111111111111111111111"),
        nc(-134217729, "-134217729", "-0x8000001", "-0o1000000001", "-0b1000000000000000000000000001"),
        nc(-134217728, "-134217728", "-0x8000000", "-0o1000000000", "-0b1000000000000000000000000000"),
        nc(-134217727, "-134217727", "-0x7ffffff", "-0o777777777", "-0b111111111111111111111111111"),
        nc(-100000001, "-100000001", "-0x5f5e101", "-0o575360401", "-0b101111101011110000100000001"),
        nc(-100000000, "-100000000", "-0x5f5e100", "-0o575360400", "-0b101111101011110000100000000"),
        nc(-99999999, "-99999999", "-0x5f5e0ff", "-0o575360377", "-0b101111101011110000011111111"),
        nc(-92233720, "-92233720", "-0x57f5ff8", "-0o537657770", "-0b101011111110101111111111000"),
        nc(-67108865, "-67108865", "-0x4000001", "-0o400000001", "-0b100000000000000000000000001"),
        nc(-67108864, "-67108864", "-0x4000000", "-0o400000000", "-0b100000000000000000000000000"),
        nc(-67108863, "-67108863", "-0x3ffffff", "-0o377777777", "-0b11111111111111111111111111"),
        nc(-33554433, "-33554433", "-0x2000001", "-0o200000001", "-0b10000000000000000000000001"),
        nc(-33554432, "-33554432", "-0x2000000", "-0o200000000", "-0b10000000000000000000000000"),
        nc(-33554431, "-33554431", "-0x1ffffff", "-0o177777777", "-0b1111111111111111111111111"),
        nc(-16777217, "-16777217", "-0x1000001", "-0o100000001", "-0b1000000000000000000000001"),
        nc(-16777216, "-16777216", "-0x1000000", "-0o100000000", "-0b1000000000000000000000000"),
        nc(-16777215, "-16777215", "-0xffffff", "-0o77777777", "-0b111111111111111111111111"),
        nc(-10000001, "-10000001", "-0x989681", "-0o46113201", "-0b100110001001011010000001"),
        nc(-10000000, "-10000000", "-0x989680", "-0o46113200", "-0b100110001001011010000000"),
        nc(-9999999, "-9999999", "-0x98967f", "-0o46113177", "-0b100110001001011001111111"),
        nc(-9223372, "-9223372", "-0x8cbccc", "-0o43136314", "-0b100011001011110011001100"),
        nc(-8388609, "-8388609", "-0x800001", "-0o40000001", "-0b100000000000000000000001"),
        nc(-8388608, "-8388608", "-0x800000", "-0o40000000", "-0b100000000000000000000000"),
        nc(-8388607, "-8388607", "-0x7fffff", "-0o37777777", "-0b11111111111111111111111"),
        nc(-4194305, "-4194305", "-0x400001", "-0o20000001", "-0b10000000000000000000001"),
        nc(-4194304, "-4194304", "-0x400000", "-0o20000000", "-0b10000000000000000000000"),
        nc(-4194303, "-4194303", "-0x3fffff", "-0o17777777", "-0b1111111111111111111111"),
        nc(-2097153, "-2097153", "-0x200001", "-0o10000001", "-0b1000000000000000000001"),
        nc(-2097152, "-2097152", "-0x200000", "-0o10000000", "-0b1000000000000000000000"),
        nc(-2097151, "-2097151", "-0x1fffff", "-0o7777777", "-0b111111111111111111111"),
        nc(-1048577, "-1048577", "-0x100001", "-0o4000001", "-0b100000000000000000001"),
        nc(-1048576, "-1048576", "-0x100000", "-0o4000000", "-0b100000000000000000000"),
        nc(-1048575, "-1048575", "-0xfffff", "-0o3777777", "-0b11111111111111111111"),
        nc(-1000001, "-1000001", "-0xf4241", "-0o3641101", "-0b11110100001001000001"),
        nc(-1000000, "-1000000", "-0xf4240", "-0o3641100", "-0b11110100001001000000"),
        nc(-999999, "-999999", "-0xf423f", "-0o3641077", "-0b11110100001000111111"),
        nc(-922337, "-922337", "-0xe12e1", "-0o3411341", "-0b11100001001011100001"),
        nc(-524289, "-524289", "-0x80001", "-0o2000001", "-0b10000000000000000001"),
        nc(-524288, "-524288", "-0x80000", "-0o2000000", "-0b10000000000000000000"),
        nc(-524287, "-524287", "-0x7ffff", "-0o1777777", "-0b1111111111111111111"),
        nc(-262145, "-262145", "-0x40001", "-0o1000001", "-0b1000000000000000001"),
        nc(-262144, "-262144", "-0x40000", "-0o1000000", "-0b1000000000000000000"),
        nc(-262143, "-262143", "-0x3ffff", "-0o777777", "-0b111111111111111111"),
        nc(-131073, "-131073", "-0x20001", "-0o400001", "-0b100000000000000001"),
        nc(-131072, "-131072", "-0x20000", "-0o400000", "-0b100000000000000000"),
        nc(-131071, "-131071", "-0x1ffff", "-0o377777", "-0b11111111111111111"),
        nc(-100001, "-100001", "-0x186a1", "-0o303241", "-0b11000011010100001"),
        nc(-100000, "-100000", "-0x186a0", "-0o303240", "-0b11000011010100000"),
        nc(-99999, "-99999", "-0x1869f", "-0o303237", "-0b11000011010011111"),
        nc(-92233, "-92233", "-0x16849", "-0o264111", "-0b10110100001001001"),
        nc(-65537, "-65537", "-0x10001", "-0o200001", "-0b10000000000000001"),
        nc(-65536, "-65536", "-0x10000", "-0o200000", "-0b10000000000000000"),
        nc(-65535, "-65535", "-0xffff", "-0o177777", "-0b1111111111111111"),
        nc(-32769, "-32769", "-0x8001", "-0o100001", "-0b1000000000000001"),
        nc(-32768, "-32768", "-0x8000", "-0o100000", "-0b1000000000000000"),
        nc(-32767, "-32767", "-0x7fff", "-0o77777", "-0b111111111111111"),
        nc(-16385, "-16385", "-0x4001", "-0o40001", "-0b100000000000001"),
        nc(-16384, "-16384", "-0x4000", "-0o40000", "-0b100000000000000"),
        nc(-16383, "-16383", "-0x3fff", "-0o37777", "-0b11111111111111"),
        nc(-10001, "-10001", "-0x2711", "-0o23421", "-0b10011100010001"),
        nc(-10000, "-10000", "-0x2710", "-0o23420", "-0b10011100010000"),
        nc(-9999, "-9999", "-0x270f", "-0o23417", "-0b10011100001111"),
        nc(-9223, "-9223", "-0x2407", "-0o22007", "-0b10010000000111"),
        nc(-8193, "-8193", "-0x2001", "-0o20001", "-0b10000000000001"),
        nc(-8192, "-8192", "-0x2000", "-0o20000", "-0b10000000000000"),
        nc(-8191, "-8191", "-0x1fff", "-0o17777", "-0b1111111111111"),
        nc(-4097, "-4097", "-0x1001", "-0o10001", "-0b1000000000001"),
        nc(-4096, "-4096", "-0x1000", "-0o10000", "-0b1000000000000"),
        nc(-4095, "-4095", "-0xfff", "-0o7777", "-0b111111111111"),
        nc(-2049, "-2049", "-0x801", "-0o4001", "-0b100000000001"),
        nc(-2048, "-2048", "-0x800", "-0o4000", "-0b100000000000"),
        nc(-2047, "-2047", "-0x7ff", "-0o3777", "-0b11111111111"),
        nc(-1025, "-1025", "-0x401", "-0o2001", "-0b10000000001"),
        nc(-1024, "-1024", "-0x400", "-0o2000", "-0b10000000000"),
        nc(-1023, "-1023", "-0x3ff", "-0o1777", "-0b1111111111"),
        nc(-1001, "-1001", "-0x3e9", "-0o1751", "-0b1111101001"),
        nc(-1000, "-1000", "-0x3e8", "-0o1750", "-0b1111101000"),
        nc(-999, "-999", "-0x3e7", "-0o1747", "-0b1111100111"),
        nc(-922, "-922", "-0x39a", "-0o1632", "-0b1110011010"),
        nc(-513, "-513", "-0x201", "-0o1001", "-0b1000000001"),
        nc(-512, "-512", "-0x200", "-0o1000", "-0b1000000000"),
        nc(-511, "-511", "-0x1ff", "-0o777", "-0b111111111"),
        nc(-257, "-257", "-0x101", "-0o401", "-0b100000001"),
        nc(-256, "-256", "-0x100", "-0o400", "-0b100000000"),
        nc(-255, "-255", "-0xff", "-0o377", "-0b11111111"),
        nc(-129, "-129", "-0x81", "-0o201", "-0b10000001"),
        nc(-128, "-128", "-0x80", "-0o200", "-0b10000000"),
        nc(-127, "-127", "-0x7f", "-0o177", "-0b1111111"),
        nc(-101, "-101", "-0x65", "-0o145", "-0b1100101"),
        nc(-100, "-100", "-0x64", "-0o144", "-0b1100100"),
        nc(-99, "-99", "-0x63", "-0o143", "-0b1100011"),
        nc(-92, "-92", "-0x5c", "-0o134", "-0b1011100"),
        nc(-65, "-65", "-0x41", "-0o101", "-0b1000001"),
        nc(-64, "-64", "-0x40", "-0o100", "-0b1000000"),
        nc(-63, "-63", "-0x3f", "-0o77", "-0b111111"),
        nc(-33, "-33", "-0x21", "-0o41", "-0b100001"),
        nc(-32, "-32", "-0x20", "-0o40", "-0b100000"),
        nc(-31, "-31", "-0x1f", "-0o37", "-0b11111"),
        nc(-17, "-17", "-0x11", "-0o21", "-0b10001"),
        nc(-16, "-16", "-0x10", "-0o20", "-0b10000"),
        nc(-15, "-15", "-0xf", "-0o17", "-0b1111"),
        nc(-11, "-11", "-0xb", "-0o13", "-0b1011"),
        nc(-10, "-10", "-0xa", "-0o12", "-0b1010"),
        nc(-9, "-9", "-0x9", "-0o11", "-0b1001"),
        nc(-8, "-8", "-0x8", "-0o10", "-0b1000"),
        nc(-7, "-7", "-0x7", "-0o7", "-0b111"),
        nc(-6, "-6", "-0x6", "-0o6", "-0b110"),
        nc(-5, "-5", "-0x5", "-0o5", "-0b101"),
        nc(-4, "-4", "-0x4", "-0o4", "-0b100"),
        nc(-3, "-3", "-0x3", "-0o3", "-0b11"),
        nc(-2, "-2", "-0x2", "-0o2", "-0b10"),
        nc(-1, "-1", "-0x1", "-0o1", "-0b1"),
        nc(0, "0", "0x0", "0o0", "0b0"),
        nc(1, "1", "0x1", "0o1", "0b1"),
        nc(2, "2", "0x2", "0o2", "0b10"),
        nc(3, "3", "0x3", "0o3", "0b11"),
        nc(4, "4", "0x4", "0o4", "0b100"),
        nc(5, "5", "0x5", "0o5", "0b101"),
        nc(6, "6", "0x6", "0o6", "0b110"),
        nc(7, "7", "0x7", "0o7", "0b111"),
        nc(8, "8", "0x8", "0o10", "0b1000"),
        nc(9, "9", "0x9", "0o11", "0b1001"),
        nc(10, "10", "0xa", "0o12", "0b1010"),
        nc(11, "11", "0xb", "0o13", "0b1011"),
        nc(15, "15", "0xf", "0o17", "0b1111"),
        nc(16, "16", "0x10", "0o20", "0b10000"),
        nc(17, "17", "0x11", "0o21", "0b10001"),
        nc(31, "31", "0x1f", "0o37", "0b11111"),
        nc(32, "32", "0x20", "0o40", "0b100000"),
        nc(33, "33", "0x21", "0o41", "0b100001"),
        nc(63, "63", "0x3f", "0o77", "0b111111"),
        nc(64, "64", "0x40", "0o100", "0b1000000"),
        nc(65, "65", "0x41", "0o101", "0b1000001"),
        nc(91, "91", "0x5b", "0o133", "0b1011011"),
        nc(92, "92", "0x5c", "0o134", "0b1011100"),
        nc(93, "93", "0x5d", "0o135", "0b1011101"),
        nc(99, "99", "0x63", "0o143", "0b1100011"),
        nc(100, "100", "0x64", "0o144", "0b1100100"),
        nc(101, "101", "0x65", "0o145", "0b1100101"),
        nc(127, "127", "0x7f", "0o177", "0b1111111"),
        nc(128, "128", "0x80", "0o200", "0b10000000"),
        nc(129, "129", "0x81", "0o201", "0b10000001"),
        nc(255, "255", "0xff", "0o377", "0b11111111"),
        nc(256, "256", "0x100", "0o400", "0b100000000"),
        nc(257, "257", "0x101", "0o401", "0b100000001"),
        nc(511, "511", "0x1ff", "0o777", "0b111111111"),
        nc(512, "512", "0x200", "0o1000", "0b1000000000"),
        nc(513, "513", "0x201", "0o1001", "0b1000000001"),
        nc(921, "921", "0x399", "0o1631", "0b1110011001"),
        nc(922, "922", "0x39a", "0o1632", "0b1110011010"),
        nc(923, "923", "0x39b", "0o1633", "0b1110011011"),
        nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
        nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
        nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
        nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
        nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
        nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
        nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
        nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
        nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
        nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
        nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
        nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
        nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
        nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
        nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
        nc(9222, "9222", "0x2406", "0o22006", "0b10010000000110"),
        nc(9223, "9223", "0x2407", "0o22007", "0b10010000000111"),
        nc(9224, "9224", "0x2408", "0o22010", "0b10010000001000"),
        nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
        nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
        nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
        nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
        nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
        nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
        nc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111"),
        nc(32768, "32768", "0x8000", "0o100000", "0b1000000000000000"),
        nc(32769, "32769", "0x8001", "0o100001", "0b1000000000000001"),
        nc(65535, "65535", "0xffff", "0o177777", "0b1111111111111111"),
        nc(65536, "65536", "0x10000", "0o200000", "0b10000000000000000"),
        nc(65537, "65537", "0x10001", "0o200001", "0b10000000000000001"),
        nc(92232, "92232", "0x16848", "0o264110", "0b10110100001001000"),
        nc(92233, "92233", "0x16849", "0o264111", "0b10110100001001001"),
        nc(92234, "92234", "0x1684a", "0o264112", "0b10110100001001010"),
        nc(99999, "99999", "0x1869f", "0o303237", "0b11000011010011111"),
        nc(100000, "100000", "0x186a0", "0o303240", "0b11000011010100000"),
        nc(100001, "100001", "0x186a1", "0o303241", "0b11000011010100001"),
        nc(131071, "131071", "0x1ffff", "0o377777", "0b11111111111111111"),
        nc(131072, "131072", "0x20000", "0o400000", "0b100000000000000000"),
        nc(131073, "131073", "0x20001", "0o400001", "0b100000000000000001"),
        nc(262143, "262143", "0x3ffff", "0o777777", "0b111111111111111111"),
        nc(262144, "262144", "0x40000", "0o1000000", "0b1000000000000000000"),
        nc(262145, "262145", "0x40001", "0o1000001", "0b1000000000000000001"),
        nc(524287, "524287", "0x7ffff", "0o1777777", "0b1111111111111111111"),
        nc(524288, "524288", "0x80000", "0o2000000", "0b10000000000000000000"),
        nc(524289, "524289", "0x80001", "0o2000001", "0b10000000000000000001"),
        nc(922336, "922336", "0xe12e0", "0o3411340", "0b11100001001011100000"),
        nc(922337, "922337", "0xe12e1", "0o3411341", "0b11100001001011100001"),
        nc(922338, "922338", "0xe12e2", "0o3411342", "0b11100001001011100010"),
        nc(999999, "999999", "0xf423f", "0o3641077", "0b11110100001000111111"),
        nc(1000000, "1000000", "0xf4240", "0o3641100", "0b11110100001001000000"),
        nc(1000001, "1000001", "0xf4241", "0o3641101", "0b11110100001001000001"),
        nc(1048575, "1048575", "0xfffff", "0o3777777", "0b11111111111111111111"),
        nc(1048576, "1048576", "0x100000", "0o4000000", "0b100000000000000000000"),
        nc(1048577, "1048577", "0x100001", "0o4000001", "0b100000000000000000001"),
        nc(2097151, "2097151", "0x1fffff", "0o7777777", "0b111111111111111111111"),
        nc(2097152, "2097152", "0x200000", "0o10000000", "0b1000000000000000000000"),
        nc(2097153, "2097153", "0x200001", "0o10000001", "0b1000000000000000000001"),
        nc(4194303, "4194303", "0x3fffff", "0o17777777", "0b1111111111111111111111"),
        nc(4194304, "4194304", "0x400000", "0o20000000", "0b10000000000000000000000"),
        nc(4194305, "4194305", "0x400001", "0o20000001", "0b10000000000000000000001"),
        nc(8388607, "8388607", "0x7fffff", "0o37777777", "0b11111111111111111111111"),
        nc(8388608, "8388608", "0x800000", "0o40000000", "0b100000000000000000000000"),
        nc(8388609, "8388609", "0x800001", "0o40000001", "0b100000000000000000000001"),
        nc(9223371, "9223371", "0x8cbccb", "0o43136313", "0b100011001011110011001011"),
        nc(9223372, "9223372", "0x8cbccc", "0o43136314", "0b100011001011110011001100"),
        nc(9223373, "9223373", "0x8cbccd", "0o43136315", "0b100011001011110011001101"),
        nc(9999999, "9999999", "0x98967f", "0o46113177", "0b100110001001011001111111"),
        nc(10000000, "10000000", "0x989680", "0o46113200", "0b100110001001011010000000"),
        nc(10000001, "10000001", "0x989681", "0o46113201", "0b100110001001011010000001"),
        nc(16777215, "16777215", "0xffffff", "0o77777777", "0b111111111111111111111111"),
        nc(16777216, "16777216", "0x1000000", "0o100000000", "0b1000000000000000000000000"),
        nc(16777217, "16777217", "0x1000001", "0o100000001", "0b1000000000000000000000001"),
        nc(33554431, "33554431", "0x1ffffff", "0o177777777", "0b1111111111111111111111111"),
        nc(33554432, "33554432", "0x2000000", "0o200000000", "0b10000000000000000000000000"),
        nc(33554433, "33554433", "0x2000001", "0o200000001", "0b10000000000000000000000001"),
        nc(67108863, "67108863", "0x3ffffff", "0o377777777", "0b11111111111111111111111111"),
        nc(67108864, "67108864", "0x4000000", "0o400000000", "0b100000000000000000000000000"),
        nc(67108865, "67108865", "0x4000001", "0o400000001", "0b100000000000000000000000001"),
        nc(92233719, "92233719", "0x57f5ff7", "0o537657767", "0b101011111110101111111110111"),
        nc(92233720, "92233720", "0x57f5ff8", "0o537657770", "0b101011111110101111111111000"),
        nc(92233721, "92233721", "0x57f5ff9", "0o537657771", "0b101011111110101111111111001"),
        nc(99999999, "99999999", "0x5f5e0ff", "0o575360377", "0b101111101011110000011111111"),
        nc(100000000, "100000000", "0x5f5e100", "0o575360400", "0b101111101011110000100000000"),
        nc(100000001, "100000001", "0x5f5e101", "0o575360401", "0b101111101011110000100000001"),
        nc(134217727, "134217727", "0x7ffffff", "0o777777777", "0b111111111111111111111111111"),
        nc(134217728, "134217728", "0x8000000", "0o1000000000", "0b1000000000000000000000000000"),
        nc(134217729, "134217729", "0x8000001", "0o1000000001", "0b1000000000000000000000000001"),
        nc(268435455, "268435455", "0xfffffff", "0o1777777777", "0b1111111111111111111111111111"),
        nc(268435456, "268435456", "0x10000000", "0o2000000000", "0b10000000000000000000000000000"),
        nc(268435457, "268435457", "0x10000001", "0o2000000001", "0b10000000000000000000000000001"),
        nc(536870911, "536870911", "0x1fffffff", "0o3777777777", "0b11111111111111111111111111111"),
        nc(536870912, "536870912", "0x20000000", "0o4000000000", "0b100000000000000000000000000000"),
        nc(536870913, "536870913", "0x20000001", "0o4000000001", "0b100000000000000000000000000001"),
        nc(922337202, "922337202", "0x36f9bfb2", "0o6676337662", "0b110110111110011011111110110010"),
        nc(922337203, "922337203", "0x36f9bfb3", "0o6676337663", "0b110110111110011011111110110011"),
        nc(922337204, "922337204", "0x36f9bfb4", "0o6676337664", "0b110110111110011011111110110100"),
        nc(999999999, "999999999", "0x3b9ac9ff", "0o7346544777", "0b111011100110101100100111111111"),
        nc(1000000000, "1000000000", "0x3b9aca00", "0o7346545000", "0b111011100110101100101000000000"),
        nc(1000000001, "1000000001", "0x3b9aca01", "0o7346545001", "0b111011100110101100101000000001"),
        nc(1073741823, "1073741823", "0x3fffffff", "0o7777777777", "0b111111111111111111111111111111"),
        nc(1073741824, "1073741824", "0x40000000", "0o10000000000", "0b1000000000000000000000000000000"),
        nc(1073741825, "1073741825", "0x40000001", "0o10000000001", "0b1000000000000000000000000000001"),
        nc(2147483647, "2147483647", "0x7fffffff", "0o17777777777", "0b1111111111111111111111111111111"),
        nc(2147483648, "2147483648", "0x80000000", "0o20000000000", "0b10000000000000000000000000000000"),
        nc(2147483649, "2147483649", "0x80000001", "0o20000000001", "0b10000000000000000000000000000001"),
        nc(4294967295, "4294967295", "0xffffffff", "0o37777777777", "0b11111111111111111111111111111111"),
        nc(4294967296, "4294967296", "0x100000000", "0o40000000000", "0b100000000000000000000000000000000"),
        nc(4294967297, "4294967297", "0x100000001", "0o40000000001", "0b100000000000000000000000000000001"),
        nc(8589934591, "8589934591", "0x1ffffffff", "0o77777777777", "0b111111111111111111111111111111111"),
        nc(8589934592, "8589934592", "0x200000000", "0o100000000000", "0b1000000000000000000000000000000000"),
        nc(8589934593, "8589934593", "0x200000001", "0o100000000001", "0b1000000000000000000000000000000001"),
        nc(9223372035, "9223372035", "0x225c17d03", "0o104560276403", "0b1000100101110000010111110100000011"),
        nc(9223372036, "9223372036", "0x225c17d04", "0o104560276404", "0b1000100101110000010111110100000100"),
        nc(9223372037, "9223372037", "0x225c17d05", "0o104560276405", "0b1000100101110000010111110100000101"),
        nc(9999999999, "9999999999", "0x2540be3ff", "0o112402761777", "0b1001010100000010111110001111111111"),
        nc(10000000000, "10000000000", "0x2540be400", "0o112402762000", "0b1001010100000010111110010000000000"),
        nc(10000000001, "10000000001", "0x2540be401", "0o112402762001", "0b1001010100000010111110010000000001"),
        nc(17179869183, "17179869183", "0x3ffffffff", "0o177777777777", "0b1111111111111111111111111111111111"),
        nc(17179869184, "17179869184", "0x400000000", "0o200000000000", "0b10000000000000000000000000000000000"),
        nc(17179869185, "17179869185", "0x400000001", "0o200000000001", "0b10000000000000000000000000000000001"),
        nc(34359738367, "34359738367", "0x7ffffffff", "0o377777777777", "0b11111111111111111111111111111111111"),
        nc(34359738368, "34359738368", "0x800000000", "0o400000000000", "0b100000000000000000000000000000000000"),
        nc(34359738369, "34359738369", "0x800000001", "0o400000000001", "0b100000000000000000000000000000000001"),
        nc(68719476735, "68719476735", "0xfffffffff", "0o777777777777", "0b111111111111111111111111111111111111"),
        nc(68719476736, "68719476736", "0x1000000000", "0o1000000000000", "0b1000000000000000000000000000000000000"),
        nc(68719476737, "68719476737", "0x1000000001", "0o1000000000001", "0b1000000000000000000000000000000000001"),
        nc(92233720367, "92233720367", "0x15798ee22f", "0o1257143561057", "0b1010101111001100011101110001000101111"),
        nc(92233720368, "92233720368", "0x15798ee230", "0o1257143561060", "0b1010101111001100011101110001000110000"),
        nc(92233720369, "92233720369", "0x15798ee231", "0o1257143561061", "0b1010101111001100011101110001000110001"),
        nc(99999999999, "99999999999", "0x174876e7ff", "0o1351035563777", "0b1011101001000011101101110011111111111"),
        nc(100000000000, "100000000000", "0x174876e800", "0o1351035564000", "0b1011101001000011101101110100000000000"),
        nc(100000000001, "100000000001", "0x174876e801", "0o1351035564001", "0b1011101001000011101101110100000000001"),
        nc(137438953471, "137438953471", "0x1fffffffff", "0o1777777777777", "0b1111111111111111111111111111111111111"),
        nc(137438953472, "137438953472", "0x2000000000", "0o2000000000000", "0b10000000000000000000000000000000000000"),
        nc(137438953473, "137438953473", "0x2000000001", "0o2000000000001", "0b10000000000000000000000000000000000001"),
        nc(274877906943, "274877906943", "0x3fffffffff", "0o3777777777777", "0b11111111111111111111111111111111111111"),
        nc(274877906944, "274877906944", "0x4000000000", "0o4000000000000", "0b100000000000000000000000000000000000000"),
        nc(274877906945, "274877906945", "0x4000000001", "0o4000000000001", "0b100000000000000000000000000000000000001"),
        nc(549755813887, "549755813887", "0x7fffffffff", "0o7777777777777", "0b111111111111111111111111111111111111111"),
        nc(549755813888, "549755813888", "0x8000000000", "0o10000000000000", "0b1000000000000000000000000000000000000000"),
        nc(549755813889, "549755813889", "0x8000000001", "0o10000000000001", "0b1000000000000000000000000000000000000001"),
        nc(922337203684, "922337203684", "0xd6bf94d5e4", "0o15327745152744", "0b1101011010111111100101001101010111100100"),
        nc(922337203685, "922337203685", "0xd6bf94d5e5", "0o15327745152745", "0b1101011010111111100101001101010111100101"),
        nc(922337203686, "922337203686", "0xd6bf94d5e6", "0o15327745152746", "0b1101011010111111100101001101010111100110"),
        nc(999999999999, "999999999999", "0xe8d4a50fff", "0o16432451207777", "0b1110100011010100101001010000111111111111"),
        nc(1000000000000, "1000000000000", "0xe8d4a51000", "0o16432451210000", "0b1110100011010100101001010001000000000000"),
        nc(1000000000001, "1000000000001", "0xe8d4a51001", "0o16432451210001", "0b1110100011010100101001010001000000000001"),
        nc(1099511627775, "1099511627775", "0xffffffffff", "0o17777777777777", "0b1111111111111111111111111111111111111111"),
        nc(1099511627776, "1099511627776", "0x10000000000", "0o20000000000000", "0b10000000000000000000000000000000000000000"),
        nc(1099511627777, "1099511627777", "0x10000000001", "0o20000000000001", "0b10000000000000000000000000000000000000001"),
        nc(2199023255551, "2199023255551", "0x1ffffffffff", "0o37777777777777", "0b11111111111111111111111111111111111111111"),
        nc(2199023255552, "2199023255552", "0x20000000000", "0o40000000000000", "0b100000000000000000000000000000000000000000"),
        nc(2199023255553, "2199023255553", "0x20000000001", "0o40000000000001", "0b100000000000000000000000000000000000000001"),
        nc(4398046511103, "4398046511103", "0x3ffffffffff", "0o77777777777777", "0b111111111111111111111111111111111111111111"),
        nc(4398046511104, "4398046511104", "0x40000000000", "0o100000000000000", "0b1000000000000000000000000000000000000000000"),
        nc(4398046511105, "4398046511105", "0x40000000001", "0o100000000000001", "0b1000000000000000000000000000000000000000001"),
        nc(8796093022207, "8796093022207", "0x7ffffffffff", "0o177777777777777", "0b1111111111111111111111111111111111111111111"),
        nc(8796093022208, "8796093022208", "0x80000000000", "0o200000000000000", "0b10000000000000000000000000000000000000000000"),
        nc(8796093022209, "8796093022209", "0x80000000001", "0o200000000000001", "0b10000000000000000000000000000000000000000001"),
        nc(9223372036853, "9223372036853", "0x8637bd05af5", "0o206157364055365", "0b10000110001101111011110100000101101011110101"),
        nc(9223372036854, "9223372036854", "0x8637bd05af6", "0o206157364055366", "0b10000110001101111011110100000101101011110110"),
        nc(9223372036855, "9223372036855", "0x8637bd05af7", "0o206157364055367", "0b10000110001101111011110100000101101011110111"),
        nc(9999999999999, "9999999999999", "0x9184e729fff", "0o221411634517777", "0b10010001100001001110011100101001111111111111"),
        nc(10000000000000, "10000000000000", "0x9184e72a000", "0o221411634520000", "0b10010001100001001110011100101010000000000000"),
        nc(10000000000001, "10000000000001", "0x9184e72a001", "0o221411634520001", "0b10010001100001001110011100101010000000000001"),
        nc(17592186044415, "17592186044415", "0xfffffffffff", "0o377777777777777", "0b11111111111111111111111111111111111111111111"),
        nc(17592186044416, "17592186044416", "0x100000000000", "0o400000000000000", "0b100000000000000000000000000000000000000000000"),
        nc(17592186044417, "17592186044417", "0x100000000001", "0o400000000000001", "0b100000000000000000000000000000000000000000001"),
        nc(35184372088831, "35184372088831", "0x1fffffffffff", "0o777777777777777", "0b111111111111111111111111111111111111111111111"),
        nc(35184372088832, "35184372088832", "0x200000000000", "0o1000000000000000", "0b1000000000000000000000000000000000000000000000"),
        nc(35184372088833, "35184372088833", "0x200000000001", "0o1000000000000001", "0b1000000000000000000000000000000000000000000001"),
        nc(70368744177663, "70368744177663", "0x3fffffffffff", "0o1777777777777777", "0b1111111111111111111111111111111111111111111111"),
        nc(70368744177664, "70368744177664", "0x400000000000", "0o2000000000000000", "0b10000000000000000000000000000000000000000000000"),
        nc(70368744177665, "70368744177665", "0x400000000001", "0o2000000000000001", "0b10000000000000000000000000000000000000000000001"),
        nc(92233720368546, "92233720368546", "0x53e2d6238da2", "0o2476132610706642", "0b10100111110001011010110001000111000110110100010"),
        nc(92233720368547, "92233720368547", "0x53e2d6238da3", "0o2476132610706643", "0b10100111110001011010110001000111000110110100011"),
        nc(92233720368548, "92233720368548", "0x53e2d6238da4", "0o2476132610706644", "0b10100111110001011010110001000111000110110100100"),
        nc(99999999999999, "99999999999999", "0x5af3107a3fff", "0o2657142036437777", "0b10110101111001100010000011110100011111111111111"),
        nc(100000000000000, "100000000000000", "0x5af3107a4000", "0o2657142036440000", "0b10110101111001100010000011110100100000000000000"),
        nc(100000000000001, "100000000000001", "0x5af3107a4001", "0o2657142036440001", "0b10110101111001100010000011110100100000000000001"),
        nc(140737488355327, "140737488355327", "0x7fffffffffff", "0o3777777777777777", "0b11111111111111111111111111111111111111111111111"),
        nc(140737488355328, "140737488355328", "0x800000000000", "0o4000000000000000", "0b100000000000000000000000000000000000000000000000"),
        nc(140737488355329, "140737488355329", "0x800000000001", "0o4000000000000001", "0b100000000000000000000000000000000000000000000001"),
        nc(281474976710655, "281474976710655", "0xffffffffffff", "0o7777777777777777", "0b111111111111111111111111111111111111111111111111"),
        nc(281474976710656, "281474976710656", "0x1000000000000", "0o10000000000000000", "0b1000000000000000000000000000000000000000000000000"),
        nc(281474976710657, "281474976710657", "0x1000000000001", "0o10000000000000001", "0b1000000000000000000000000000000000000000000000001"),
        nc(562949953421311, "562949953421311", "0x1ffffffffffff", "0o17777777777777777", "0b1111111111111111111111111111111111111111111111111"),
        nc(562949953421312, "562949953421312", "0x2000000000000", "0o20000000000000000", "0b10000000000000000000000000000000000000000000000000"),
        nc(562949953421313, "562949953421313", "0x2000000000001", "0o20000000000000001", "0b10000000000000000000000000000000000000000000000001"),
        nc(922337203685476, "922337203685476", "0x346dc5d638864", "0o32155613530704144", "0b11010001101101110001011101011000111000100001100100"),
        nc(922337203685477, "922337203685477", "0x346dc5d638865", "0o32155613530704145", "0b11010001101101110001011101011000111000100001100101"),
        nc(922337203685478, "922337203685478", "0x346dc5d638866", "0o32155613530704146", "0b11010001101101110001011101011000111000100001100110"),
        nc(999999999999999, "999999999999999", "0x38d7ea4c67fff", "0o34327724461477777", "0b11100011010111111010100100110001100111111111111111"),
        nc(1000000000000000, "1000000000000000", "0x38d7ea4c68000", "0o34327724461500000", "0b11100011010111111010100100110001101000000000000000"),
        nc(1000000000000001, "1000000000000001", "0x38d7ea4c68001", "0o34327724461500001", "0b11100011010111111010100100110001101000000000000001"),
        nc(1125899906842623, "1125899906842623", "0x3ffffffffffff", "0o37777777777777777", "0b11111111111111111111111111111111111111111111111111"),
        nc(1125899906842624, "1125899906842624", "0x4000000000000", "0o40000000000000000", "0b100000000000000000000000000000000000000000000000000"),
        nc(1125899906842625, "1125899906842625", "0x4000000000001", "0o40000000000000001", "0b100000000000000000000000000000000000000000000000001"),
        nc(2251799813685247, "2251799813685247", "0x7ffffffffffff", "0o77777777777777777", "0b111111111111111111111111111111111111111111111111111"),
        nc(2251799813685248, "2251799813685248", "0x8000000000000", "0o100000000000000000", "0b1000000000000000000000000000000000000000000000000000"),
        nc(2251799813685249, "2251799813685249", "0x8000000000001", "0o100000000000000001", "0b1000000000000000000000000000000000000000000000000001"),
        nc(4503599627370495, "4503599627370495", "0xfffffffffffff", "0o177777777777777777", "0b1111111111111111111111111111111111111111111111111111"),
        nc(4503599627370496, "4503599627370496", "0x10000000000000", "0o200000000000000000", "0b10000000000000000000000000000000000000000000000000000"),
        nc(4503599627370497, "4503599627370497", "0x10000000000001", "0o200000000000000001", "0b10000000000000000000000000000000000000000000000000001"),
        nc(9007199254740991, "9007199254740991", "0x1fffffffffffff", "0o377777777777777777", "0b11111111111111111111111111111111111111111111111111111"),
        nc(9007199254740992, "9007199254740992", "0x20000000000000", "0o400000000000000000", "0b100000000000000000000000000000000000000000000000000000"),
        nc(9007199254740993, "9007199254740993", "0x20000000000001", "0o400000000000000001", "0b100000000000000000000000000000000000000000000000000001"),
        nc(9223372036854775, "9223372036854775", "0x20c49ba5e353f7", "0o406111564570651767", "0b100000110001001001101110100101111000110101001111110111"),
        nc(9223372036854776, "9223372036854776", "0x20c49ba5e353f8", "0o406111564570651770", "0b100000110001001001101110100101111000110101001111111000"),
        nc(9223372036854777, "9223372036854777", "0x20c49ba5e353f9", "0o406111564570651771", "0b100000110001001001101110100101111000110101001111111001"),
        nc(9999999999999999, "9999999999999999", "0x2386f26fc0ffff", "0o434157115760177777", "0b100011100001101111001001101111110000001111111111111111"),
        nc(10000000000000000, "10000000000000000", "0x2386f26fc10000", "0o434157115760200000", "0b100011100001101111001001101111110000010000000000000000"),
        nc(10000000000000001, "10000000000000001", "0x2386f26fc10001", "0o434157115760200001", "0b100011100001101111001001101111110000010000000000000001"),
        nc(18014398509481983, "18014398509481983", "0x3fffffffffffff", "0o777777777777777777", "0b111111111111111111111111111111111111111111111111111111"),
        nc(18014398509481984, "18014398509481984", "0x40000000000000", "0o1000000000000000000", "0b1000000000000000000000000000000000000000000000000000000"),
        nc(18014398509481985, "18014398509481985", "0x40000000000001", "0o1000000000000000001", "0b1000000000000000000000000000000000000000000000000000001"),
        nc(36028797018963967, "36028797018963967", "0x7fffffffffffff", "0o1777777777777777777", "0b1111111111111111111111111111111111111111111111111111111"),
        nc(36028797018963968, "36028797018963968", "0x80000000000000", "0o2000000000000000000", "0b10000000000000000000000000000000000000000000000000000000"),
        nc(36028797018963969, "36028797018963969", "0x80000000000001", "0o2000000000000000001", "0b10000000000000000000000000000000000000000000000000000001"),
        nc(72057594037927935, "72057594037927935", "0xffffffffffffff", "0o3777777777777777777", "0b11111111111111111111111111111111111111111111111111111111"),
        nc(72057594037927936, "72057594037927936", "0x100000000000000", "0o4000000000000000000", "0b100000000000000000000000000000000000000000000000000000000"),
        nc(72057594037927937, "72057594037927937", "0x100000000000001", "0o4000000000000000001", "0b100000000000000000000000000000000000000000000000000000001"),
        nc(92233720368547759, "92233720368547759", "0x147ae147ae147af", "0o5075341217270243657", "0b101000111101011100001010001111010111000010100011110101111"),
        nc(92233720368547760, "92233720368547760", "0x147ae147ae147b0", "0o5075341217270243660", "0b101000111101011100001010001111010111000010100011110110000"),
        nc(92233720368547761, "92233720368547761", "0x147ae147ae147b1", "0o5075341217270243661", "0b101000111101011100001010001111010111000010100011110110001"),
        nc(99999999999999999, "99999999999999999", "0x16345785d89ffff", "0o5432127413542377777", "0b101100011010001010111100001011101100010011111111111111111"),
        nc(100000000000000000, "100000000000000000", "0x16345785d8a0000", "0o5432127413542400000", "0b101100011010001010111100001011101100010100000000000000000"),
        nc(100000000000000001, "100000000000000001", "0x16345785d8a0001", "0o5432127413542400001", "0b101100011010001010111100001011101100010100000000000000001"),
        nc(144115188075855871, "144115188075855871", "0x1ffffffffffffff", "0o7777777777777777777", "0b111111111111111111111111111111111111111111111111111111111"),
        nc(144115188075855872, "144115188075855872", "0x200000000000000", "0o10000000000000000000", "0b1000000000000000000000000000000000000000000000000000000000"),
        nc(144115188075855873, "144115188075855873", "0x200000000000001", "0o10000000000000000001", "0b1000000000000000000000000000000000000000000000000000000001"),
        nc(288230376151711743, "288230376151711743", "0x3ffffffffffffff", "0o17777777777777777777", "0b1111111111111111111111111111111111111111111111111111111111"),
        nc(288230376151711744, "288230376151711744", "0x400000000000000", "0o20000000000000000000", "0b10000000000000000000000000000000000000000000000000000000000"),
        nc(288230376151711745, "288230376151711745", "0x400000000000001", "0o20000000000000000001", "0b10000000000000000000000000000000000000000000000000000000001"),
        nc(576460752303423487, "576460752303423487", "0x7ffffffffffffff", "0o37777777777777777777", "0b11111111111111111111111111111111111111111111111111111111111"),
        nc(576460752303423488, "576460752303423488", "0x800000000000000", "0o40000000000000000000", "0b100000000000000000000000000000000000000000000000000000000000"),
        nc(576460752303423489, "576460752303423489", "0x800000000000001", "0o40000000000000000001", "0b100000000000000000000000000000000000000000000000000000000001"),
        nc(922337203685477631, "922337203685477631", "0xcccccccccccccff", "0o63146314631463146377", "0b110011001100110011001100110011001100110011001100110011111111"),
        nc(922337203685477632, "922337203685477632", "0xccccccccccccd00", "0o63146314631463146400", "0b110011001100110011001100110011001100110011001100110100000000"),
        nc(922337203685477633, "922337203685477633", "0xccccccccccccd01", "0o63146314631463146401", "0b110011001100110011001100110011001100110011001100110100000001"),
        nc(999999999999999999, "999999999999999999", "0xde0b6b3a763ffff", "0o67405553164730777777", "0b110111100000101101101011001110100111011000111111111111111111"),
        nc(1000000000000000000, "1000000000000000000", "0xde0b6b3a7640000", "0o67405553164731000000", "0b110111100000101101101011001110100111011001000000000000000000"),
        nc(1000000000000000001, "1000000000000000001", "0xde0b6b3a7640001", "0o67405553164731000001", "0b110111100000101101101011001110100111011001000000000000000001"),
        nc(1152921504606846975, "1152921504606846975", "0xfffffffffffffff", "0o77777777777777777777", "0b111111111111111111111111111111111111111111111111111111111111"),
        nc(1152921504606846976, "1152921504606846976", "0x1000000000000000", "0o100000000000000000000", "0b1000000000000000000000000000000000000000000000000000000000000"),
        nc(1152921504606846977, "1152921504606846977", "0x1000000000000001", "0o100000000000000000001", "0b1000000000000000000000000000000000000000000000000000000000001"),
        nc(2305843009213693951, "2305843009213693951", "0x1fffffffffffffff", "0o177777777777777777777", "0b1111111111111111111111111111111111111111111111111111111111111"),
        nc(2305843009213693952, "2305843009213693952", "0x2000000000000000", "0o200000000000000000000", "0b10000000000000000000000000000000000000000000000000000000000000"),
        nc(2305843009213693953, "2305843009213693953", "0x2000000000000001", "0o200000000000000000001", "0b10000000000000000000000000000000000000000000000000000000000001"),
        nc(4611686018427387903, "4611686018427387903", "0x3fffffffffffffff", "0o377777777777777777777", "0b11111111111111111111111111111111111111111111111111111111111111"),
        nc(4611686018427387904, "4611686018427387904", "0x4000000000000000", "0o400000000000000000000", "0b100000000000000000000000000000000000000000000000000000000000000"),
        nc(4611686018427387905, "4611686018427387905", "0x4000000000000001", "0o400000000000000000001", "0b100000000000000000000000000000000000000000000000000000000000001"),
        nc(9223372036854775802, "9223372036854775802", "0x7ffffffffffffffa", "0o777777777777777777772", "0b111111111111111111111111111111111111111111111111111111111111010"),
        nc(9223372036854775803, "9223372036854775803", "0x7ffffffffffffffb", "0o777777777777777777773", "0b111111111111111111111111111111111111111111111111111111111111011"),
        nc(9223372036854775804, "9223372036854775804", "0x7ffffffffffffffc", "0o777777777777777777774", "0b111111111111111111111111111111111111111111111111111111111111100"),
        pc(9223372036854775805, "9223372036854775805", "0x7ffffffffffffffd", "0o777777777777777777775", "0b111111111111111111111111111111111111111111111111111111111111101", 9223372036854775806),
        pc(9223372036854775806, "9223372036854775806", "0x7ffffffffffffffe", "0o777777777777777777776", "0b111111111111111111111111111111111111111111111111111111111111110", 9223372036854775807),
        pcm1(9223372036854775807, "9223372036854775807", "0x7fffffffffffffff", "0o777777777777777777777", "0b111111111111111111111111111111111111111111111111111111111111111", -9223372036854775807),
        #undef nc
        #undef pc
    };
};


template<>
struct numbers<uint64_t>
{
    using value_type = uint64_t;
    static C4_INLINE_CONSTEXPR const number_case<uint64_t> vals[] = {
        #define nc(val, dec, hex, bin, oct) nc_(UINT64_C, val, dec, hex, bin, oct)
        #define pc(val, dec, hex, bin, oct, valp1) ncp_(UINT64_C, val, dec, hex, bin, oct, valp1)
        pc(0, "0", "0x0", "0o0", "0b0", 1),
        nc(1, "1", "0x1", "0o1", "0b1"),
        nc(2, "2", "0x2", "0o2", "0b10"),
        nc(3, "3", "0x3", "0o3", "0b11"),
        nc(4, "4", "0x4", "0o4", "0b100"),
        nc(5, "5", "0x5", "0o5", "0b101"),
        nc(6, "6", "0x6", "0o6", "0b110"),
        nc(7, "7", "0x7", "0o7", "0b111"),
        nc(8, "8", "0x8", "0o10", "0b1000"),
        nc(9, "9", "0x9", "0o11", "0b1001"),
        nc(10, "10", "0xa", "0o12", "0b1010"),
        nc(11, "11", "0xb", "0o13", "0b1011"),
        nc(15, "15", "0xf", "0o17", "0b1111"),
        nc(16, "16", "0x10", "0o20", "0b10000"),
        nc(17, "17", "0x11", "0o21", "0b10001"),
        nc(18, "18", "0x12", "0o22", "0b10010"),
        nc(19, "19", "0x13", "0o23", "0b10011"),
        nc(31, "31", "0x1f", "0o37", "0b11111"),
        nc(32, "32", "0x20", "0o40", "0b100000"),
        nc(33, "33", "0x21", "0o41", "0b100001"),
        nc(63, "63", "0x3f", "0o77", "0b111111"),
        nc(64, "64", "0x40", "0o100", "0b1000000"),
        nc(65, "65", "0x41", "0o101", "0b1000001"),
        nc(99, "99", "0x63", "0o143", "0b1100011"),
        nc(100, "100", "0x64", "0o144", "0b1100100"),
        nc(101, "101", "0x65", "0o145", "0b1100101"),
        nc(127, "127", "0x7f", "0o177", "0b1111111"),
        nc(128, "128", "0x80", "0o200", "0b10000000"),
        nc(129, "129", "0x81", "0o201", "0b10000001"),
        nc(183, "183", "0xb7", "0o267", "0b10110111"),
        nc(184, "184", "0xb8", "0o270", "0b10111000"),
        nc(185, "185", "0xb9", "0o271", "0b10111001"),
        nc(255, "255", "0xff", "0o377", "0b11111111"),
        nc(256, "256", "0x100", "0o400", "0b100000000"),
        nc(257, "257", "0x101", "0o401", "0b100000001"),
        nc(511, "511", "0x1ff", "0o777", "0b111111111"),
        nc(512, "512", "0x200", "0o1000", "0b1000000000"),
        nc(513, "513", "0x201", "0o1001", "0b1000000001"),
        nc(999, "999", "0x3e7", "0o1747", "0b1111100111"),
        nc(1000, "1000", "0x3e8", "0o1750", "0b1111101000"),
        nc(1001, "1001", "0x3e9", "0o1751", "0b1111101001"),
        nc(1023, "1023", "0x3ff", "0o1777", "0b1111111111"),
        nc(1024, "1024", "0x400", "0o2000", "0b10000000000"),
        nc(1025, "1025", "0x401", "0o2001", "0b10000000001"),
        nc(1843, "1843", "0x733", "0o3463", "0b11100110011"),
        nc(1844, "1844", "0x734", "0o3464", "0b11100110100"),
        nc(1845, "1845", "0x735", "0o3465", "0b11100110101"),
        nc(2047, "2047", "0x7ff", "0o3777", "0b11111111111"),
        nc(2048, "2048", "0x800", "0o4000", "0b100000000000"),
        nc(2049, "2049", "0x801", "0o4001", "0b100000000001"),
        nc(4095, "4095", "0xfff", "0o7777", "0b111111111111"),
        nc(4096, "4096", "0x1000", "0o10000", "0b1000000000000"),
        nc(4097, "4097", "0x1001", "0o10001", "0b1000000000001"),
        nc(8191, "8191", "0x1fff", "0o17777", "0b1111111111111"),
        nc(8192, "8192", "0x2000", "0o20000", "0b10000000000000"),
        nc(8193, "8193", "0x2001", "0o20001", "0b10000000000001"),
        nc(9999, "9999", "0x270f", "0o23417", "0b10011100001111"),
        nc(10000, "10000", "0x2710", "0o23420", "0b10011100010000"),
        nc(10001, "10001", "0x2711", "0o23421", "0b10011100010001"),
        nc(16383, "16383", "0x3fff", "0o37777", "0b11111111111111"),
        nc(16384, "16384", "0x4000", "0o40000", "0b100000000000000"),
        nc(16385, "16385", "0x4001", "0o40001", "0b100000000000001"),
        nc(18445, "18445", "0x480d", "0o44015", "0b100100000001101"),
        nc(18446, "18446", "0x480e", "0o44016", "0b100100000001110"),
        nc(18447, "18447", "0x480f", "0o44017", "0b100100000001111"),
        nc(32767, "32767", "0x7fff", "0o77777", "0b111111111111111"),
        nc(32768, "32768", "0x8000", "0o100000", "0b1000000000000000"),
        nc(32769, "32769", "0x8001", "0o100001", "0b1000000000000001"),
        nc(65535, "65535", "0xffff", "0o177777", "0b1111111111111111"),
        nc(65536, "65536", "0x10000", "0o200000", "0b10000000000000000"),
        nc(65537, "65537", "0x10001", "0o200001", "0b10000000000000001"),
        nc(99999, "99999", "0x1869f", "0o303237", "0b11000011010011111"),
        nc(100000, "100000", "0x186a0", "0o303240", "0b11000011010100000"),
        nc(100001, "100001", "0x186a1", "0o303241", "0b11000011010100001"),
        nc(131071, "131071", "0x1ffff", "0o377777", "0b11111111111111111"),
        nc(131072, "131072", "0x20000", "0o400000", "0b100000000000000000"),
        nc(131073, "131073", "0x20001", "0o400001", "0b100000000000000001"),
        nc(184466, "184466", "0x2d092", "0o550222", "0b101101000010010010"),
        nc(184467, "184467", "0x2d093", "0o550223", "0b101101000010010011"),
        nc(184468, "184468", "0x2d094", "0o550224", "0b101101000010010100"),
        nc(262143, "262143", "0x3ffff", "0o777777", "0b111111111111111111"),
        nc(262144, "262144", "0x40000", "0o1000000", "0b1000000000000000000"),
        nc(262145, "262145", "0x40001", "0o1000001", "0b1000000000000000001"),
        nc(524287, "524287", "0x7ffff", "0o1777777", "0b1111111111111111111"),
        nc(524288, "524288", "0x80000", "0o2000000", "0b10000000000000000000"),
        nc(524289, "524289", "0x80001", "0o2000001", "0b10000000000000000001"),
        nc(999999, "999999", "0xf423f", "0o3641077", "0b11110100001000111111"),
        nc(1000000, "1000000", "0xf4240", "0o3641100", "0b11110100001001000000"),
        nc(1000001, "1000001", "0xf4241", "0o3641101", "0b11110100001001000001"),
        nc(1048575, "1048575", "0xfffff", "0o3777777", "0b11111111111111111111"),
        nc(1048576, "1048576", "0x100000", "0o4000000", "0b100000000000000000000"),
        nc(1048577, "1048577", "0x100001", "0o4000001", "0b100000000000000000001"),
        nc(1844673, "1844673", "0x1c25c1", "0o7022701", "0b111000010010111000001"),
        nc(1844674, "1844674", "0x1c25c2", "0o7022702", "0b111000010010111000010"),
        nc(1844675, "1844675", "0x1c25c3", "0o7022703", "0b111000010010111000011"),
        nc(2097151, "2097151", "0x1fffff", "0o7777777", "0b111111111111111111111"),
        nc(2097152, "2097152", "0x200000", "0o10000000", "0b1000000000000000000000"),
        nc(2097153, "2097153", "0x200001", "0o10000001", "0b1000000000000000000001"),
        nc(4194303, "4194303", "0x3fffff", "0o17777777", "0b1111111111111111111111"),
        nc(4194304, "4194304", "0x400000", "0o20000000", "0b10000000000000000000000"),
        nc(4194305, "4194305", "0x400001", "0o20000001", "0b10000000000000000000001"),
        nc(8388607, "8388607", "0x7fffff", "0o37777777", "0b11111111111111111111111"),
        nc(8388608, "8388608", "0x800000", "0o40000000", "0b100000000000000000000000"),
        nc(8388609, "8388609", "0x800001", "0o40000001", "0b100000000000000000000001"),
        nc(9999999, "9999999", "0x98967f", "0o46113177", "0b100110001001011001111111"),
        nc(10000000, "10000000", "0x989680", "0o46113200", "0b100110001001011010000000"),
        nc(10000001, "10000001", "0x989681", "0o46113201", "0b100110001001011010000001"),
        nc(16777215, "16777215", "0xffffff", "0o77777777", "0b111111111111111111111111"),
        nc(16777216, "16777216", "0x1000000", "0o100000000", "0b1000000000000000000000000"),
        nc(16777217, "16777217", "0x1000001", "0o100000001", "0b1000000000000000000000001"),
        nc(18446743, "18446743", "0x1197997", "0o106274627", "0b1000110010111100110010111"),
        nc(18446744, "18446744", "0x1197998", "0o106274630", "0b1000110010111100110011000"),
        nc(18446745, "18446745", "0x1197999", "0o106274631", "0b1000110010111100110011001"),
        nc(33554431, "33554431", "0x1ffffff", "0o177777777", "0b1111111111111111111111111"),
        nc(33554432, "33554432", "0x2000000", "0o200000000", "0b10000000000000000000000000"),
        nc(33554433, "33554433", "0x2000001", "0o200000001", "0b10000000000000000000000001"),
        nc(67108863, "67108863", "0x3ffffff", "0o377777777", "0b11111111111111111111111111"),
        nc(67108864, "67108864", "0x4000000", "0o400000000", "0b100000000000000000000000000"),
        nc(67108865, "67108865", "0x4000001", "0o400000001", "0b100000000000000000000000001"),
        nc(99999999, "99999999", "0x5f5e0ff", "0o575360377", "0b101111101011110000011111111"),
        nc(100000000, "100000000", "0x5f5e100", "0o575360400", "0b101111101011110000100000000"),
        nc(100000001, "100000001", "0x5f5e101", "0o575360401", "0b101111101011110000100000001"),
        nc(134217727, "134217727", "0x7ffffff", "0o777777777", "0b111111111111111111111111111"),
        nc(134217728, "134217728", "0x8000000", "0o1000000000", "0b1000000000000000000000000000"),
        nc(134217729, "134217729", "0x8000001", "0o1000000001", "0b1000000000000000000000000001"),
        nc(184467439, "184467439", "0xafebfef", "0o1277537757", "0b1010111111101011111111101111"),
        nc(184467440, "184467440", "0xafebff0", "0o1277537760", "0b1010111111101011111111110000"),
        nc(184467441, "184467441", "0xafebff1", "0o1277537761", "0b1010111111101011111111110001"),
        nc(268435455, "268435455", "0xfffffff", "0o1777777777", "0b1111111111111111111111111111"),
        nc(268435456, "268435456", "0x10000000", "0o2000000000", "0b10000000000000000000000000000"),
        nc(268435457, "268435457", "0x10000001", "0o2000000001", "0b10000000000000000000000000001"),
        nc(536870911, "536870911", "0x1fffffff", "0o3777777777", "0b11111111111111111111111111111"),
        nc(536870912, "536870912", "0x20000000", "0o4000000000", "0b100000000000000000000000000000"),
        nc(536870913, "536870913", "0x20000001", "0o4000000001", "0b100000000000000000000000000001"),
        nc(999999999, "999999999", "0x3b9ac9ff", "0o7346544777", "0b111011100110101100100111111111"),
        nc(1000000000, "1000000000", "0x3b9aca00", "0o7346545000", "0b111011100110101100101000000000"),
        nc(1000000001, "1000000001", "0x3b9aca01", "0o7346545001", "0b111011100110101100101000000001"),
        nc(1073741823, "1073741823", "0x3fffffff", "0o7777777777", "0b111111111111111111111111111111"),
        nc(1073741824, "1073741824", "0x40000000", "0o10000000000", "0b1000000000000000000000000000000"),
        nc(1073741825, "1073741825", "0x40000001", "0o10000000001", "0b1000000000000000000000000000001"),
        nc(1844674406, "1844674406", "0x6df37f66", "0o15574677546", "0b1101101111100110111111101100110"),
        nc(1844674407, "1844674407", "0x6df37f67", "0o15574677547", "0b1101101111100110111111101100111"),
        nc(1844674408, "1844674408", "0x6df37f68", "0o15574677550", "0b1101101111100110111111101101000"),
        nc(2147483647, "2147483647", "0x7fffffff", "0o17777777777", "0b1111111111111111111111111111111"),
        nc(2147483648, "2147483648", "0x80000000", "0o20000000000", "0b10000000000000000000000000000000"),
        nc(2147483649, "2147483649", "0x80000001", "0o20000000001", "0b10000000000000000000000000000001"),
        nc(4294967295, "4294967295", "0xffffffff", "0o37777777777", "0b11111111111111111111111111111111"),
        nc(4294967296, "4294967296", "0x100000000", "0o40000000000", "0b100000000000000000000000000000000"),
        nc(4294967297, "4294967297", "0x100000001", "0o40000000001", "0b100000000000000000000000000000001"),
        nc(8589934591, "8589934591", "0x1ffffffff", "0o77777777777", "0b111111111111111111111111111111111"),
        nc(8589934592, "8589934592", "0x200000000", "0o100000000000", "0b1000000000000000000000000000000000"),
        nc(8589934593, "8589934593", "0x200000001", "0o100000000001", "0b1000000000000000000000000000000001"),
        nc(9999999999, "9999999999", "0x2540be3ff", "0o112402761777", "0b1001010100000010111110001111111111"),
        nc(10000000000, "10000000000", "0x2540be400", "0o112402762000", "0b1001010100000010111110010000000000"),
        nc(10000000001, "10000000001", "0x2540be401", "0o112402762001", "0b1001010100000010111110010000000001"),
        nc(17179869183, "17179869183", "0x3ffffffff", "0o177777777777", "0b1111111111111111111111111111111111"),
        nc(17179869184, "17179869184", "0x400000000", "0o200000000000", "0b10000000000000000000000000000000000"),
        nc(17179869185, "17179869185", "0x400000001", "0o200000000001", "0b10000000000000000000000000000000001"),
        nc(18446744072, "18446744072", "0x44b82fa08", "0o211340575010", "0b10001001011100000101111101000001000"),
        nc(18446744073, "18446744073", "0x44b82fa09", "0o211340575011", "0b10001001011100000101111101000001001"),
        nc(18446744074, "18446744074", "0x44b82fa0a", "0o211340575012", "0b10001001011100000101111101000001010"),
        nc(34359738367, "34359738367", "0x7ffffffff", "0o377777777777", "0b11111111111111111111111111111111111"),
        nc(34359738368, "34359738368", "0x800000000", "0o400000000000", "0b100000000000000000000000000000000000"),
        nc(34359738369, "34359738369", "0x800000001", "0o400000000001", "0b100000000000000000000000000000000001"),
        nc(68719476735, "68719476735", "0xfffffffff", "0o777777777777", "0b111111111111111111111111111111111111"),
        nc(68719476736, "68719476736", "0x1000000000", "0o1000000000000", "0b1000000000000000000000000000000000000"),
        nc(68719476737, "68719476737", "0x1000000001", "0o1000000000001", "0b1000000000000000000000000000000000001"),
        nc(99999999999, "99999999999", "0x174876e7ff", "0o1351035563777", "0b1011101001000011101101110011111111111"),
        nc(100000000000, "100000000000", "0x174876e800", "0o1351035564000", "0b1011101001000011101101110100000000000"),
        nc(100000000001, "100000000001", "0x174876e801", "0o1351035564001", "0b1011101001000011101101110100000000001"),
        nc(137438953471, "137438953471", "0x1fffffffff", "0o1777777777777", "0b1111111111111111111111111111111111111"),
        nc(137438953472, "137438953472", "0x2000000000", "0o2000000000000", "0b10000000000000000000000000000000000000"),
        nc(137438953473, "137438953473", "0x2000000001", "0o2000000000001", "0b10000000000000000000000000000000000001"),
        nc(184467440736, "184467440736", "0x2af31dc460", "0o2536307342140", "0b10101011110011000111011100010001100000"),
        nc(184467440737, "184467440737", "0x2af31dc461", "0o2536307342141", "0b10101011110011000111011100010001100001"),
        nc(184467440738, "184467440738", "0x2af31dc462", "0o2536307342142", "0b10101011110011000111011100010001100010"),
        nc(274877906943, "274877906943", "0x3fffffffff", "0o3777777777777", "0b11111111111111111111111111111111111111"),
        nc(274877906944, "274877906944", "0x4000000000", "0o4000000000000", "0b100000000000000000000000000000000000000"),
        nc(274877906945, "274877906945", "0x4000000001", "0o4000000000001", "0b100000000000000000000000000000000000001"),
        nc(549755813887, "549755813887", "0x7fffffffff", "0o7777777777777", "0b111111111111111111111111111111111111111"),
        nc(549755813888, "549755813888", "0x8000000000", "0o10000000000000", "0b1000000000000000000000000000000000000000"),
        nc(549755813889, "549755813889", "0x8000000001", "0o10000000000001", "0b1000000000000000000000000000000000000001"),
        nc(999999999999, "999999999999", "0xe8d4a50fff", "0o16432451207777", "0b1110100011010100101001010000111111111111"),
        nc(1000000000000, "1000000000000", "0xe8d4a51000", "0o16432451210000", "0b1110100011010100101001010001000000000000"),
        nc(1000000000001, "1000000000001", "0xe8d4a51001", "0o16432451210001", "0b1110100011010100101001010001000000000001"),
        nc(1099511627775, "1099511627775", "0xffffffffff", "0o17777777777777", "0b1111111111111111111111111111111111111111"),
        nc(1099511627776, "1099511627776", "0x10000000000", "0o20000000000000", "0b10000000000000000000000000000000000000000"),
        nc(1099511627777, "1099511627777", "0x10000000001", "0o20000000000001", "0b10000000000000000000000000000000000000001"),
        nc(1844674407369, "1844674407369", "0x1ad7f29abc9", "0o32657712325711", "0b11010110101111111001010011010101111001001"),
        nc(1844674407370, "1844674407370", "0x1ad7f29abca", "0o32657712325712", "0b11010110101111111001010011010101111001010"),
        nc(1844674407371, "1844674407371", "0x1ad7f29abcb", "0o32657712325713", "0b11010110101111111001010011010101111001011"),
        nc(2199023255551, "2199023255551", "0x1ffffffffff", "0o37777777777777", "0b11111111111111111111111111111111111111111"),
        nc(2199023255552, "2199023255552", "0x20000000000", "0o40000000000000", "0b100000000000000000000000000000000000000000"),
        nc(2199023255553, "2199023255553", "0x20000000001", "0o40000000000001", "0b100000000000000000000000000000000000000001"),
        nc(4398046511103, "4398046511103", "0x3ffffffffff", "0o77777777777777", "0b111111111111111111111111111111111111111111"),
        nc(4398046511104, "4398046511104", "0x40000000000", "0o100000000000000", "0b1000000000000000000000000000000000000000000"),
        nc(4398046511105, "4398046511105", "0x40000000001", "0o100000000000001", "0b1000000000000000000000000000000000000000001"),
        nc(8796093022207, "8796093022207", "0x7ffffffffff", "0o177777777777777", "0b1111111111111111111111111111111111111111111"),
        nc(8796093022208, "8796093022208", "0x80000000000", "0o200000000000000", "0b10000000000000000000000000000000000000000000"),
        nc(8796093022209, "8796093022209", "0x80000000001", "0o200000000000001", "0b10000000000000000000000000000000000000000001"),
        nc(9999999999999, "9999999999999", "0x9184e729fff", "0o221411634517777", "0b10010001100001001110011100101001111111111111"),
        nc(10000000000000, "10000000000000", "0x9184e72a000", "0o221411634520000", "0b10010001100001001110011100101010000000000000"),
        nc(10000000000001, "10000000000001", "0x9184e72a001", "0o221411634520001", "0b10010001100001001110011100101010000000000001"),
        nc(17592186044415, "17592186044415", "0xfffffffffff", "0o377777777777777", "0b11111111111111111111111111111111111111111111"),
        nc(17592186044416, "17592186044416", "0x100000000000", "0o400000000000000", "0b100000000000000000000000000000000000000000000"),
        nc(17592186044417, "17592186044417", "0x100000000001", "0o400000000000001", "0b100000000000000000000000000000000000000000001"),
        nc(18446744073708, "18446744073708", "0x10c6f7a0b5ec", "0o414336750132754", "0b100001100011011110111101000001011010111101100"),
        nc(18446744073709, "18446744073709", "0x10c6f7a0b5ed", "0o414336750132755", "0b100001100011011110111101000001011010111101101"),
        nc(18446744073710, "18446744073710", "0x10c6f7a0b5ee", "0o414336750132756", "0b100001100011011110111101000001011010111101110"),
        nc(35184372088831, "35184372088831", "0x1fffffffffff", "0o777777777777777", "0b111111111111111111111111111111111111111111111"),
        nc(35184372088832, "35184372088832", "0x200000000000", "0o1000000000000000", "0b1000000000000000000000000000000000000000000000"),
        nc(35184372088833, "35184372088833", "0x200000000001", "0o1000000000000001", "0b1000000000000000000000000000000000000000000001"),
        nc(70368744177663, "70368744177663", "0x3fffffffffff", "0o1777777777777777", "0b1111111111111111111111111111111111111111111111"),
        nc(70368744177664, "70368744177664", "0x400000000000", "0o2000000000000000", "0b10000000000000000000000000000000000000000000000"),
        nc(70368744177665, "70368744177665", "0x400000000001", "0o2000000000000001", "0b10000000000000000000000000000000000000000000001"),
        nc(99999999999999, "99999999999999", "0x5af3107a3fff", "0o2657142036437777", "0b10110101111001100010000011110100011111111111111"),
        nc(100000000000000, "100000000000000", "0x5af3107a4000", "0o2657142036440000", "0b10110101111001100010000011110100100000000000000"),
        nc(100000000000001, "100000000000001", "0x5af3107a4001", "0o2657142036440001", "0b10110101111001100010000011110100100000000000001"),
        nc(140737488355327, "140737488355327", "0x7fffffffffff", "0o3777777777777777", "0b11111111111111111111111111111111111111111111111"),
        nc(140737488355328, "140737488355328", "0x800000000000", "0o4000000000000000", "0b100000000000000000000000000000000000000000000000"),
        nc(140737488355329, "140737488355329", "0x800000000001", "0o4000000000000001", "0b100000000000000000000000000000000000000000000001"),
        nc(184467440737094, "184467440737094", "0xa7c5ac471b46", "0o5174265421615506", "0b101001111100010110101100010001110001101101000110"),
        nc(184467440737095, "184467440737095", "0xa7c5ac471b47", "0o5174265421615507", "0b101001111100010110101100010001110001101101000111"),
        nc(184467440737096, "184467440737096", "0xa7c5ac471b48", "0o5174265421615510", "0b101001111100010110101100010001110001101101001000"),
        nc(281474976710655, "281474976710655", "0xffffffffffff", "0o7777777777777777", "0b111111111111111111111111111111111111111111111111"),
        nc(281474976710656, "281474976710656", "0x1000000000000", "0o10000000000000000", "0b1000000000000000000000000000000000000000000000000"),
        nc(281474976710657, "281474976710657", "0x1000000000001", "0o10000000000000001", "0b1000000000000000000000000000000000000000000000001"),
        nc(562949953421311, "562949953421311", "0x1ffffffffffff", "0o17777777777777777", "0b1111111111111111111111111111111111111111111111111"),
        nc(562949953421312, "562949953421312", "0x2000000000000", "0o20000000000000000", "0b10000000000000000000000000000000000000000000000000"),
        nc(562949953421313, "562949953421313", "0x2000000000001", "0o20000000000000001", "0b10000000000000000000000000000000000000000000000001"),
        nc(999999999999999, "999999999999999", "0x38d7ea4c67fff", "0o34327724461477777", "0b11100011010111111010100100110001100111111111111111"),
        nc(1000000000000000, "1000000000000000", "0x38d7ea4c68000", "0o34327724461500000", "0b11100011010111111010100100110001101000000000000000"),
        nc(1000000000000001, "1000000000000001", "0x38d7ea4c68001", "0o34327724461500001", "0b11100011010111111010100100110001101000000000000001"),
        nc(1125899906842623, "1125899906842623", "0x3ffffffffffff", "0o37777777777777777", "0b11111111111111111111111111111111111111111111111111"),
        nc(1125899906842624, "1125899906842624", "0x4000000000000", "0o40000000000000000", "0b100000000000000000000000000000000000000000000000000"),
        nc(1125899906842625, "1125899906842625", "0x4000000000001", "0o40000000000000001", "0b100000000000000000000000000000000000000000000000001"),
        nc(1844674407370954, "1844674407370954", "0x68db8bac710ca", "0o64333427261610312", "0b110100011011011100010111010110001110001000011001010"),
        nc(1844674407370955, "1844674407370955", "0x68db8bac710cb", "0o64333427261610313", "0b110100011011011100010111010110001110001000011001011"),
        nc(1844674407370956, "1844674407370956", "0x68db8bac710cc", "0o64333427261610314", "0b110100011011011100010111010110001110001000011001100"),
        nc(2251799813685247, "2251799813685247", "0x7ffffffffffff", "0o77777777777777777", "0b111111111111111111111111111111111111111111111111111"),
        nc(2251799813685248, "2251799813685248", "0x8000000000000", "0o100000000000000000", "0b1000000000000000000000000000000000000000000000000000"),
        nc(2251799813685249, "2251799813685249", "0x8000000000001", "0o100000000000000001", "0b1000000000000000000000000000000000000000000000000001"),
        nc(4503599627370495, "4503599627370495", "0xfffffffffffff", "0o177777777777777777", "0b1111111111111111111111111111111111111111111111111111"),
        nc(4503599627370496, "4503599627370496", "0x10000000000000", "0o200000000000000000", "0b10000000000000000000000000000000000000000000000000000"),
        nc(4503599627370497, "4503599627370497", "0x10000000000001", "0o200000000000000001", "0b10000000000000000000000000000000000000000000000000001"),
        nc(9007199254740991, "9007199254740991", "0x1fffffffffffff", "0o377777777777777777", "0b11111111111111111111111111111111111111111111111111111"),
        nc(9007199254740992, "9007199254740992", "0x20000000000000", "0o400000000000000000", "0b100000000000000000000000000000000000000000000000000000"),
        nc(9007199254740993, "9007199254740993", "0x20000000000001", "0o400000000000000001", "0b100000000000000000000000000000000000000000000000000001"),
        nc(9999999999999999, "9999999999999999", "0x2386f26fc0ffff", "0o434157115760177777", "0b100011100001101111001001101111110000001111111111111111"),
        nc(10000000000000000, "10000000000000000", "0x2386f26fc10000", "0o434157115760200000", "0b100011100001101111001001101111110000010000000000000000"),
        nc(10000000000000001, "10000000000000001", "0x2386f26fc10001", "0o434157115760200001", "0b100011100001101111001001101111110000010000000000000001"),
        nc(18014398509481983, "18014398509481983", "0x3fffffffffffff", "0o777777777777777777", "0b111111111111111111111111111111111111111111111111111111"),
        nc(18014398509481984, "18014398509481984", "0x40000000000000", "0o1000000000000000000", "0b1000000000000000000000000000000000000000000000000000000"),
        nc(18014398509481985, "18014398509481985", "0x40000000000001", "0o1000000000000000001", "0b1000000000000000000000000000000000000000000000000000001"),
        nc(18446744073709551, "18446744073709551", "0x4189374bc6a7ef", "0o1014223351361523757", "0b1000001100010010011011101001011110001101010011111101111"),
        nc(18446744073709552, "18446744073709552", "0x4189374bc6a7f0", "0o1014223351361523760", "0b1000001100010010011011101001011110001101010011111110000"),
        nc(18446744073709553, "18446744073709553", "0x4189374bc6a7f1", "0o1014223351361523761", "0b1000001100010010011011101001011110001101010011111110001"),
        nc(36028797018963967, "36028797018963967", "0x7fffffffffffff", "0o1777777777777777777", "0b1111111111111111111111111111111111111111111111111111111"),
        nc(36028797018963968, "36028797018963968", "0x80000000000000", "0o2000000000000000000", "0b10000000000000000000000000000000000000000000000000000000"),
        nc(36028797018963969, "36028797018963969", "0x80000000000001", "0o2000000000000000001", "0b10000000000000000000000000000000000000000000000000000001"),
        nc(72057594037927935, "72057594037927935", "0xffffffffffffff", "0o3777777777777777777", "0b11111111111111111111111111111111111111111111111111111111"),
        nc(72057594037927936, "72057594037927936", "0x100000000000000", "0o4000000000000000000", "0b100000000000000000000000000000000000000000000000000000000"),
        nc(72057594037927937, "72057594037927937", "0x100000000000001", "0o4000000000000000001", "0b100000000000000000000000000000000000000000000000000000001"),
        nc(99999999999999999, "99999999999999999", "0x16345785d89ffff", "0o5432127413542377777", "0b101100011010001010111100001011101100010011111111111111111"),
        nc(100000000000000000, "100000000000000000", "0x16345785d8a0000", "0o5432127413542400000", "0b101100011010001010111100001011101100010100000000000000000"),
        nc(100000000000000001, "100000000000000001", "0x16345785d8a0001", "0o5432127413542400001", "0b101100011010001010111100001011101100010100000000000000001"),
        nc(144115188075855871, "144115188075855871", "0x1ffffffffffffff", "0o7777777777777777777", "0b111111111111111111111111111111111111111111111111111111111"),
        nc(144115188075855872, "144115188075855872", "0x200000000000000", "0o10000000000000000000", "0b1000000000000000000000000000000000000000000000000000000000"),
        nc(144115188075855873, "144115188075855873", "0x200000000000001", "0o10000000000000000001", "0b1000000000000000000000000000000000000000000000000000000001"),
        nc(184467440737095519, "184467440737095519", "0x28f5c28f5c28f5f", "0o12172702436560507537", "0b1010001111010111000010100011110101110000101000111101011111"),
        nc(184467440737095520, "184467440737095520", "0x28f5c28f5c28f60", "0o12172702436560507540", "0b1010001111010111000010100011110101110000101000111101100000"),
        nc(184467440737095521, "184467440737095521", "0x28f5c28f5c28f61", "0o12172702436560507541", "0b1010001111010111000010100011110101110000101000111101100001"),
        nc(288230376151711743, "288230376151711743", "0x3ffffffffffffff", "0o17777777777777777777", "0b1111111111111111111111111111111111111111111111111111111111"),
        nc(288230376151711744, "288230376151711744", "0x400000000000000", "0o20000000000000000000", "0b10000000000000000000000000000000000000000000000000000000000"),
        nc(288230376151711745, "288230376151711745", "0x400000000000001", "0o20000000000000000001", "0b10000000000000000000000000000000000000000000000000000000001"),
        nc(576460752303423487, "576460752303423487", "0x7ffffffffffffff", "0o37777777777777777777", "0b11111111111111111111111111111111111111111111111111111111111"),
        nc(576460752303423488, "576460752303423488", "0x800000000000000", "0o40000000000000000000", "0b100000000000000000000000000000000000000000000000000000000000"),
        nc(576460752303423489, "576460752303423489", "0x800000000000001", "0o40000000000000000001", "0b100000000000000000000000000000000000000000000000000000000001"),
        nc(999999999999999999, "999999999999999999", "0xde0b6b3a763ffff", "0o67405553164730777777", "0b110111100000101101101011001110100111011000111111111111111111"),
        nc(1000000000000000000, "1000000000000000000", "0xde0b6b3a7640000", "0o67405553164731000000", "0b110111100000101101101011001110100111011001000000000000000000"),
        nc(1000000000000000001, "1000000000000000001", "0xde0b6b3a7640001", "0o67405553164731000001", "0b110111100000101101101011001110100111011001000000000000000001"),
        nc(1152921504606846975, "1152921504606846975", "0xfffffffffffffff", "0o77777777777777777777", "0b111111111111111111111111111111111111111111111111111111111111"),
        nc(1152921504606846976, "1152921504606846976", "0x1000000000000000", "0o100000000000000000000", "0b1000000000000000000000000000000000000000000000000000000000000"),
        nc(1152921504606846977, "1152921504606846977", "0x1000000000000001", "0o100000000000000000001", "0b1000000000000000000000000000000000000000000000000000000000001"),
        nc(1844674407370955263, "1844674407370955263", "0x19999999999999ff", "0o146314631463146314777", "0b1100110011001100110011001100110011001100110011001100111111111"),
        nc(1844674407370955264, "1844674407370955264", "0x1999999999999a00", "0o146314631463146315000", "0b1100110011001100110011001100110011001100110011001101000000000"),
        nc(1844674407370955265, "1844674407370955265", "0x1999999999999a01", "0o146314631463146315001", "0b1100110011001100110011001100110011001100110011001101000000001"),
        nc(2305843009213693951, "2305843009213693951", "0x1fffffffffffffff", "0o177777777777777777777", "0b1111111111111111111111111111111111111111111111111111111111111"),
        nc(2305843009213693952, "2305843009213693952", "0x2000000000000000", "0o200000000000000000000", "0b10000000000000000000000000000000000000000000000000000000000000"),
        nc(2305843009213693953, "2305843009213693953", "0x2000000000000001", "0o200000000000000000001", "0b10000000000000000000000000000000000000000000000000000000000001"),
        nc(4611686018427387903, "4611686018427387903", "0x3fffffffffffffff", "0o377777777777777777777", "0b11111111111111111111111111111111111111111111111111111111111111"),
        nc(4611686018427387904, "4611686018427387904", "0x4000000000000000", "0o400000000000000000000", "0b100000000000000000000000000000000000000000000000000000000000000"),
        nc(4611686018427387905, "4611686018427387905", "0x4000000000000001", "0o400000000000000000001", "0b100000000000000000000000000000000000000000000000000000000000001"),
        nc(9223372036854775807, "9223372036854775807", "0x7fffffffffffffff", "0o777777777777777777777", "0b111111111111111111111111111111111111111111111111111111111111111"),
        nc(9223372036854775808, "9223372036854775808", "0x8000000000000000", "0o1000000000000000000000", "0b1000000000000000000000000000000000000000000000000000000000000000"),
        nc(9223372036854775809, "9223372036854775809", "0x8000000000000001", "0o1000000000000000000001", "0b1000000000000000000000000000000000000000000000000000000000000001"),
        nc(9999999999999999999, "9999999999999999999", "0x8ac7230489e7ffff", "0o1053071060221171777777", "0b1000101011000111001000110000010010001001111001111111111111111111"),
        nc(10000000000000000000, "10000000000000000000", "0x8ac7230489e80000", "0o1053071060221172000000", "0b1000101011000111001000110000010010001001111010000000000000000000"),
        nc(10000000000000000001, "10000000000000000001", "0x8ac7230489e80001", "0o1053071060221172000001", "0b1000101011000111001000110000010010001001111010000000000000000001"),
        nc(18446744073709551611, "18446744073709551611", "0xfffffffffffffffb", "0o1777777777777777777773", "0b1111111111111111111111111111111111111111111111111111111111111011"),
        nc(18446744073709551612, "18446744073709551612", "0xfffffffffffffffc", "0o1777777777777777777774", "0b1111111111111111111111111111111111111111111111111111111111111100"),
        nc(18446744073709551613, "18446744073709551613", "0xfffffffffffffffd", "0o1777777777777777777775", "0b1111111111111111111111111111111111111111111111111111111111111101"),
        pc(18446744073709551614, "18446744073709551614", "0xfffffffffffffffe", "0o1777777777777777777776", "0b1111111111111111111111111111111111111111111111111111111111111110", 18446744073709551615),
        pc(18446744073709551615, "18446744073709551615", "0xffffffffffffffff", "0o1777777777777777777777", "0b1111111111111111111111111111111111111111111111111111111111111111", 0),
        #undef nc
        #undef pc
    };
};

C4_INLINE_CONSTEXPR const number_case<int8_t> numbers<int8_t>::vals[];
C4_INLINE_CONSTEXPR const number_case<uint8_t> numbers<uint8_t>::vals[];

C4_INLINE_CONSTEXPR const number_case<int16_t> numbers<int16_t>::vals[];
C4_INLINE_CONSTEXPR const number_case<uint16_t> numbers<uint16_t>::vals[];

C4_INLINE_CONSTEXPR const number_case<int32_t> numbers<int32_t>::vals[];
C4_INLINE_CONSTEXPR const number_case<uint32_t> numbers<uint32_t>::vals[];

C4_INLINE_CONSTEXPR const number_case<int64_t> numbers<int64_t>::vals[];
C4_INLINE_CONSTEXPR const number_case<uint64_t> numbers<uint64_t>::vals[];

C4_SUPPRESS_WARNING_MSVC_POP

} // namespace c4
