#!/usr/bin/python3
"""
Entry point for dh-fortran command

Copyright (C) 2025 Alastair McKinstry <mckinstry@debian.org>
Released under the GPL-3 Gnu Public License.
"""

import click
import os
import sys
import dhfortran.compilers as cmplrs
import dhfortran.cli as cli
import dhfortran.targets as targets


@click.group()
@click.pass_context
def dh_fortran(ctx):
    cli.verbose_print("dh_fortran called")
    # ensure that ctx.obj exists and is a dict (in case `cli()` is called
    # by means other than the `if` block below)
    ctx.ensure_object(dict)


@dh_fortran.command("get_fc_flags")
@click.option("--fc", default=None, help="Fortran compiler")
# @cli.debhelper_common_args
def get_fc_flags(fc):
    """
    Returns set of flags to be set , either passed as FCFLAGS to another
    """
    cli.verbose_print("dh_fortran get_fc_flags called")
    cli.validate_compiler(fc)
    click.echo(cmplrs.get_fc_flags(fc))


@dh_fortran.command("get_f77_flags")
@click.option("--f77", default=None, help="Fortran compiler")
def get_f77_flags(f77):
    """
    Returns flags for a FORTRAN 77 compiler (recommended contents of FFLAGS)
    """
    cli.verbose_print("dh_fortran get_f77_flags called")
    cli.validate_compiler(f77)
    click.echo(cmplrs.get_f77_flags(f77))


@dh_fortran.command("get_fc_default", help="Returns default Fortran flavor")
@click.pass_context
def get_fc_default(ctx):
    """
    Details on get_fc_default
    """
    cli.verbose_print("dh_fortran get_fc_default called")
    click.echo(cmplrs.get_fc_default())


@dh_fortran.command("get_fc_optional", help="Returns list of optional Fortran flavors")
@click.pass_context
def get_fc_optional(ctx):
    cli.verbose_print("dh_fortran get_fc_optional called")
    click.echo(cmplrs.get_fc_optional())


@dh_fortran.command(
    "get_fc_exe", help="Return Fortran executable for specified (or default) flavor"
)
@click.option("--flavor", help="Fortran flavor (eg gfortran-15, flang-21)")
@click.pass_context
def get_fc_exe(ctx, flavor):
    cli.verbose_print("dh_fortran get_fc_exe called")
    cli.validate_flavor(flavor)
    click.echo(cmplrs.get_fc_exe(flavor))


@dh_fortran.command("get_abi_vendor", help="Returns ABI name for a given compiler")
@click.option("--flavor", help="Fortran flavor (eg gfortran-15, flang-21)")
@click.pass_context
def get_abi_vendor(ctx, flavor):
    cli.verbose_print("dh_fortran get_abi_vendor called")
    cli.validate_flavor(flavor)
    return cmplrs.get_abi_vendor(flavor)


@dh_fortran.command(
    "get_env",
    help="Return set of environment flags to execute  debian/rules targets with",
)
@click.option("--flavor", help="Fortran flavor (eg gfortran-15, flang-21)")
def get_env(flavor: str):
    cli.verbose_print("dh_fortran get_env called")
    cli.validate_flavor(flavor)
    click.echo(cmplrs.get_env(flavor))


##
## debian/rules targets
##


@cli.debhelper_common_args
@dh_fortran.command("clean", help="Execute Fortran-specific clean actions")
@click.option("--fc-default", "-d", help="Act on named package")
@click.option("--fc-optional", help="Act on named package")
def clean(fc_default, fc_optional):
    breakpoint()
    targets.clean()


@cli.debhelper_common_args
@dh_fortran.command("configure", help="Execute configure actions for each Fortran flavor")
def configure():
    targets.configure()


# Test handling of common debhelper stuff
@cli.debhelper_common_args
@dh_fortran.command("build", help="Execute build target for each Fortran flavor")
def build():
    targets.build()


@cli.debhelper_common_args
@dh_fortran.command("install", help="Execute install target for each Fortran flavor")
def install():
    targets.install()


@cli.debhelper_common_args
@dh_fortran.command("test", help="Execute test target for each Fortran flavor")
def test():
    targets.test()


if __name__ == "__main__":
    import pytest

    pytest.main(["tests/main.py"])
