/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/CrossLink.hpp"
#include "MsXpS/libXpertMassCore/Sequence.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"
#include "MsXpS/libXpertMassCore/Ionizer.hpp"
#include "MsXpS/libXpertMassCore/CalcOptions.hpp"
#include "MsXpS/libXpertMassCore/Monomer.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{

// Forward declaration
class Polymer;

class DECLSPEC Oligomer : public PropListHolder
{
  friend class Ionizer;

  public:
  Oligomer(PolymerCstQSPtr polymer_cqsp,
           const QString &name             = QString(),
           const QString &description      = QString(),
           bool is_modified                = false,
           const Ionizer &ionizer          = Ionizer(),
           const CalcOptions &calc_options = CalcOptions());

  Oligomer(const Oligomer &other);

  virtual ~Oligomer();

  //////////////// THE POLYMER /////////////////////
  void setPolymerCstSPtr(PolymerCstQSPtr polymer_cqsp);
  const PolymerCstQSPtr &getPolymerCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  QString getName() const;

  //////////////// THE DESCRIPTION /////////////////////
  void setDescription(const QString &description);
  QString getDescription() const;

  //////////////// MODIFICATION STATUS /////////////////////
  void setModified(bool is_modified = true);
  bool isModified(bool deep = false) const;
  std::size_t modifiedMonomerCount() const;

  //////////////// THE IONIZER /////////////////////
  void setIonizer(const Ionizer &ionizer);
  const Ionizer &getIonizerCstRef() const;
  Ionizer &getIonizerRef();
  Enums::IonizationOutcome ionize();
  Enums::IonizationOutcome deionize();
  Enums::IonizationOutcome molecularMasses(double &mono, double &avg) const;

  //////////////// THE SEQUENCE RANGES /////////////////////
  void setIndexRanges(const IndexRangeCollection &index_ranges);
  void setIndexRange(const IndexRange &index_range);
  const IndexRangeCollection &getIndexRangeCollectionCstRef() const;
  IndexRangeCollection &getIndexRangeCollectionRef();
  void setStartAndStopIndices(std::size_t start_index, std::size_t stop_index);
  void setStartIndex(int start_index);
  qsizetype startIndex(bool &ok) const;
  void setStopIndex(int stop_index);
  qsizetype stopIndex(bool &ok) const;

  //////////////// THE CALCULATION OPTIONS /////////////////////
  void setCalcOptions(const CalcOptions &calc_options);
  const CalcOptions &getCalcOptionsCstRef() const;
  CalcOptions &getCalcOptionsRef();

  //////////////// THE MASSES /////////////////////
  void setMass(Enums::MassType mass_type, double mass);
  void setMasses(double mono, double avg);
  double getMass(Enums::MassType mass_type) const;
  double &getMassRef(Enums::MassType mass_type);

  //////////////// THE CROSS-LINKS /////////////////////
  const std::vector<CrossLinkSPtr> &getCrossLinksCstRef() const;
  std::vector<CrossLinkSPtr> &getCrossLinksRef();
  bool addCrossLink(CrossLinkSPtr cross_link_sp);

  //////////////// THE PARTIAL CLEAVAGE /////////////////////
  void setPartialCleavage(std::size_t partial_cleavage);
  std::size_t getPartialCleavage() const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const Formula &formula);
  void setFormula(const QString &formula_string);
  const Formula &getFormulaCstRef() const;
  Formula &getFormulaRef();

  //////////////// THE MONOMERS /////////////////////
  MonomerSPtr getLeftEndMonomerCstSPtr() const;
  MonomerSPtr getRightEndMonomerCstSPtr() const;
  MonomerSPtr getMonomerCstSPtrAt(std::size_t index) const;

  //////////////// OPERATORS /////////////////////
  using PropListHolder::operator=;
  Oligomer &operator=(const Oligomer &other);

  //////////////// ELEMENTAL CALCULATION FUNCTIONS /////////////////////
  virtual QString elementalComposition(const CalcOptions &calc_options,
                                       const Ionizer &ionizer) const;
  virtual QString elementalComposition() const;

  //////////////// MASS CALCULATION FUNCTIONS /////////////////////
  virtual bool calculateMasses();
  virtual bool calculateMasses(const CalcOptions &calc_options,
                               const Ionizer &ionizer);

  std::size_t size();

  bool encompasses(std::size_t index) const;
  bool encompasses(MonomerCstRPtr monomer_crp) const;
  bool encompasses(MonomerSPtr monomer_sp) const;

  QString toString() const;

  protected:
  PolymerCstQSPtr mcsp_polymer;

  QString m_name;
  QString m_description;

  mutable bool m_isModified = false;
  Ionizer m_ionizer;

  CalcOptions m_calcOptions;

  double m_mono = 0;
  double m_avg  = 0;

  // The container of the the cross-links that are involved in the
  // formation of the cross-linked oligomer (the CrossLinkSPtr
  // instances are shared with mcsp_polymer).
  std::vector<CrossLinkSPtr> m_crossLinks;

  std::size_t m_partialCleavage = 0;
  Formula m_formula;
};

typedef std::shared_ptr<Oligomer> OligomerSPtr;
typedef std::shared_ptr<const Oligomer> OligomerCstSPtr;


} // namespace libXpertMassCore
} // namespace MsXpS
